/*******************************************************************************
BEGIN_FILE_PROLOG:

NAME:
   PGS_CSC.h

DESCRIPTION:
   This file contains the common error handling defines, typedefs
   and function prototypes.

END_FILE_PROLOG:
*******************************************************************************/

#ifndef _PGS_CSC_H_
#define _PGS_CSC_H_

/*  include files  */

#include <PGS_SMF.h>
#include <PGS_TD.h>    /* defines spacecraft tags */
#include <PGS_CBP.h>   /* defines celestial body tags */

/* error messages */

#include <PGS_CSC_4.h>

/* logical identifiers for CSC data files */

#define UTCPOLE 10401
#define EARTHFIGURE 10402

/* coordinate system transformation flags */

#define ECItoSC          10010  
#define SCtoECI          10020
#define SCtoORB          10030
#define ORBtoSC          10040
#define EtoORB           10050 /* obsolete */
#define ORBtoE           10060 /* obsolete */
#define ECItoECR         10070
#define ECRtoECI         10080
#define GEOtoECR         10090
#define ECRtoGEO         10100


/* Input Vector Tags for ZenithAzimuth tool */

#define CB   314159265   /* generic ECR vector to very distant 
                            object (many many Earth radii)  or 
                            any "free" vector pointing away  */
#define LOOK 358979323   /* vector from SC to Look Point in ECR */

/* Day/Night Limit Tags */

#define CivilTwilight      90
#define CivilNight         96
#define NauticalNight      102
#define AstronNight        108

/* Earth Model Tags */

#define WGS84  20010
/*
        AA = 6378137.0;
        CC = 6356752.31414;
*/     

#define MERIT 20020
/*
        AA = 6378137.0;
        CC = 6356752.290;
*/

#define new_intl 20030
/*
        AA = 6378157.5;
        CC = 6356772.2;
*/


/* special data types */

typedef PGSt_integer PGSt_CSC_EarthTag;


/*     Function prototypes    */


extern PGSt_SMF_status
PGS_CSC_GreenwichHour(
    PGSt_integer,
    char [28],
    PGSt_double [],
    PGSt_double []);

extern PGSt_SMF_status
PGS_CSC_georec(
    PGSt_double, 
    PGSt_double, 
    PGSt_double, 
    PGSt_double, 
    PGSt_double, 
    PGSt_double [3]);

extern PGSt_SMF_status
PGS_CSC_nutat2(
    PGSt_double ,   
    PGSt_double *,   
    PGSt_boolean,
    PGSt_double *);

extern PGSt_SMF_status
PGS_CSC_rotat6(
    PGSt_double * ,    
    PGSt_double * ,    
    PGSt_integer     ,    
    PGSt_double * );    

extern PGSt_SMF_status
PGS_CSC_wahr2(
    PGSt_double ,    
    PGSt_double *);

extern PGSt_SMF_status
PGS_CSC_precs2(
    PGSt_double  ,
    PGSt_double  ,
    PGSt_boolean,
    PGSt_double* );

extern PGSt_SMF_status
PGS_CSC_rotat3(
    PGSt_double* ,   
    PGSt_double* ,  
    PGSt_integer ,   
    PGSt_double* ); 

extern PGSt_SMF_status
PGS_CSC_DayNight(
       PGSt_integer,
       char [28],
       PGSt_double [],
       PGSt_double [],
       PGSt_double [],
       PGSt_tag ,
       PGSt_boolean []);

extern PGSt_SMF_status
PGS_CSC_UTC_UT1Pole(
    PGSt_double ,       
    PGSt_double *,         
    PGSt_double *,        
    PGSt_double *,        
    PGSt_double *);       

extern PGSt_SMF_status
PGS_CSC_ECItoECR(
    PGSt_integer,
    char [28], 
    PGSt_double [],  
    PGSt_double [][6],
    PGSt_double [][6]);

extern PGSt_SMF_status
PGS_CSC_ECItoSC(  
    PGSt_tag ,
    PGSt_integer ,
    char [28], 
    PGSt_double [],  
    PGSt_double [][3],
    PGSt_double [][3]);

extern PGSt_SMF_status
PGS_CSC_ORBtoSC(  
    PGSt_tag ,
    PGSt_integer ,
    char [28], 
    PGSt_double [],  
    PGSt_double [][3],
    PGSt_double [][3]);

extern PGSt_SMF_status
PGS_CSC_SCtoORB(  
    PGSt_tag ,
    PGSt_integer ,
    char [28], 
    PGSt_double [],  
    PGSt_double [][3],
    PGSt_double [][3]);

extern PGSt_SMF_status
PGS_CSC_EulertoOrbit_Quat(
    PGSt_double [3],
    PGSt_integer [3],
    PGSt_double [4]);

extern PGSt_SMF_status
PGS_CSC_SCtoECI(  
    PGSt_tag ,
    PGSt_integer ,
    char [28], 
    PGSt_double [],  
    PGSt_double [][3],
    PGSt_double [][3]);

extern PGSt_SMF_status
PGS_CSC_ECRtoECI(
    PGSt_integer,
    char [28],   
    PGSt_double [],      
    PGSt_double [][6],                  
    PGSt_double [][6]);

extern PGSt_SMF_status
PGS_CSC_ECRtoGEO(
    PGSt_double [3],
    char *,
    PGSt_double *,
    PGSt_double *,
    PGSt_double *);

extern PGSt_SMF_status
PGS_CSC_GEOtoECR(
    PGSt_double ,
    PGSt_double ,
    PGSt_double ,
    char *,
    PGSt_double [3]);

extern PGSt_SMF_status
PGS_CSC_GreenwichHour(
    PGSt_integer ,         
    char [28], 
    PGSt_double [],  
    PGSt_double []); 

extern PGSt_SMF_status
PGS_CSC_quatRotate(
    PGSt_double [4],
    PGSt_double [3],
    PGSt_double [3]);

extern PGSt_SMF_status
PGS_CSC_SubSatPoint(
    PGSt_tag,
    PGSt_integer,
    char [28],
    PGSt_double [], 
    char *,
    PGSt_boolean,
    PGSt_double [],      
    PGSt_double [],      
    PGSt_double [],    
    PGSt_double [][3]);     

extern PGSt_SMF_status
PGS_CSC_SubSatPointVel(
    PGSt_double [6],
    char *,
    PGSt_boolean,
    PGSt_double *,
    PGSt_double *,
    PGSt_double *,
    PGSt_double [3]);

   
extern PGSt_SMF_status
PGS_CSC_GetFOV_Pixel(
    PGSt_tag,         /* spacecraftTag */
    PGSt_integer,     /* numOffsets */
    char [28],        /* UTC */
    PGSt_double [],   /* time offsets */
    char [20],        /* Earth tag */
    PGSt_boolean,     /* accuracy Flag */
    PGSt_double [][3],/* pixel Unit vectors in SC coords */    
    PGSt_double [][3],/* instrument XYZ offset in SC coords */
    PGSt_double [],   /* latitudes  */
    PGSt_double [],   /* longitudes */
    PGSt_double [][3],/* ECR pixel unit vectors */
    PGSt_double [],   /* slant ranges */
    PGSt_double []);  /* Doppler velocities */

extern PGSt_SMF_status
PGS_CSC_GetEarthFigure(
    char *,
    PGSt_double *,
    PGSt_double *);

extern PGSt_SMF_status
PGS_CSC_LookPoint(
    PGSt_double [3],
    PGSt_double [3],
    PGSt_double,
    PGSt_double,
    PGSt_double,
    PGSt_double *,
    PGSt_double [3]);

extern PGSt_SMF_status
PGS_CSC_ZenithAzimuth( 
    PGSt_double [3],
    PGSt_double,
    PGSt_double,
    PGSt_double,
    PGSt_tag, 
    PGSt_boolean,
    PGSt_boolean, 
    PGSt_double *,
    PGSt_double *,
    PGSt_double *);

extern PGSt_SMF_status
PGS_CSC_SpaceRefract(
PGSt_double,
PGSt_double,
PGSt_double,
PGSt_double *,
PGSt_double *);

extern PGSt_double
PGS_CSC_Norm(                        
    PGSt_double *); 

extern PGSt_SMF_status
PGS_CSC_getQuats(
    PGSt_double [3][3],
    PGSt_double [4]);

extern PGSt_SMF_status
PGS_CSC_QuatToEuler(                      
    PGSt_double [],     
    PGSt_integer [],
    PGSt_double []);     

extern PGSt_SMF_status
PGS_CSC_EulerToQuat(                      
    PGSt_double [],     
    PGSt_integer [],
    PGSt_double []);     

extern PGSt_double *
PGS_CSC_crossProduct(           
    PGSt_double [],     
    PGSt_double [],       
    PGSt_double []);

extern PGSt_double
PGS_CSC_dotProduct(                
    PGSt_double [],        
    PGSt_double [],       
    PGSt_integer);        

extern PGSt_double *
PGS_CSC_getECItoORBquat(
    PGSt_double [3],   
    PGSt_double [3],   
    PGSt_double [4]);  

extern PGSt_SMF_status
PGS_CSC_getORBtoTIPquat(         
    char [28],    
    char *,
    PGSt_double [3],  
    PGSt_double [4]); 

extern PGSt_SMF_status
PGS_CSC_quatMultiply(
    PGSt_double [4],
    PGSt_double [4],
    PGSt_double [4]);

extern PGSt_SMF_status
PGS_CSC_TiltYaw(
    PGSt_double [3],
    PGSt_double [3],
    PGSt_double,
    PGSt_double,
    PGSt_double [4]);

extern PGSt_SMF_status
PGS_CSC_PointInFOVgeom(                
    PGSt_integer,       
    PGSt_double [3],       
    PGSt_double [][3],
    PGSt_double  [3],     
    PGSt_boolean *);  

extern PGSt_SMF_status
PGS_CSC_Earthpt_FOV(                   
    PGSt_integer,            
    char [28],         
    PGSt_double [],            
    PGSt_tag,        
    char  *,      
    PGSt_double,             
    PGSt_double,            
    PGSt_double,             
    PGSt_integer,                     
    PGSt_double [][3],     
    void *,    
    PGSt_boolean [],                                             
    PGSt_double [][3]);  

extern PGSt_SMF_status
PGS_CSC_FOVconicalHull(            
    PGSt_integer,   
    PGSt_double [3],  
    PGSt_double [][3],
    PGSt_double,    
    PGSt_double [3], 
    PGSt_double [][3],
    PGSt_boolean *,       
    PGSt_double  *); 

extern PGSt_SMF_status
PGS_CSC_EarthOccult(                
    PGSt_double [3],    
    PGSt_double,      
    PGSt_double,            
    PGSt_double [3],    
    PGSt_double [3],    
    PGSt_double [3],     
    PGSt_boolean *);        

/* GENERAL REFERENCES FOR ECItoECR and ECRtoECI transformations

Astronomical Almanac, published by U.S. Government Printing Office for
the U.S. Naval Observatory (annual - 1994 is referenced here).
(denoted "Astron. Almanac" or "A.A.")

Explanatory Supplement to the Astronomical Almanac
ed: Dr. P. Kenneth Seidelmann
Published by University Science Books, Mill Valley, CA 1992
(denoted "Explan. Suppl.")    */

#endif
