/* Changes made 8/96 by Michael Hensley:
 * commented out use of the file "earthfigure.dat"
 * added headed file "TS_GLOBAL.h" and the use of it's #define'd variables
 * commented out call to PGS_CSC_GetEarthFigure
 * commented out several range and error checks
 * 
 * Changes made 7/96 by Dave Bazell:
 * commented out calls to SMF routines
 * commented out #include <cfortran.h>
 */
/******************************************************************************
BEGIN_FILE_PROLOG:

FILENAME:
  PGS_CSC_GEOtoECR.c

DESCRIPTION:
  This file contains the function PGS_CSC_GEOtoECR()

AUTHOR:
  Anubha Singhal    / Applied Research Corporation
  Peter Noerdlinger / Applied Research Corporation

HISTORY:
  24-Jan-1994       PDN   Designed and created         
  16-Aug-1994       PDN   Modified design for change in calling sequence
  17-Aug-1994       AS    Original version
  31-Aug-1994       AS    Put in call to PGS_CSC_GetEarthFigure()
  15-Sep-1994       AS    Updated prologs
  21-Sep-1994       AS    Made changes from code inspection
  27-Jan-1995       AS    Fixed comment line in prolog - warning message 
                          PGSCSC_W_INVALID_ALTITUDE incorrectly specified
			  as PGSCSC_E_INVALID_ALTITUDE
  
END_FILE_PROLOG:
******************************************************************************/

/******************************************************************************
BEGIN_PROLOG:

TITLE:   
      This function converts from geodetic to ECR coordinates
 
NAME:    
      PGS_CSC_GEOtoECR() 

SYNOPSIS:
C:
      #include <PGS_CSC.h>

      PGSt_SMF_status
      PGS_CSC_GEOtoECR(
             PGSt_double       longitude,
             PGSt_double       latitude,
             PGSt_double       altitude,
             char              *earthEllipsTag,
             PGSt_double       posECR[3]);

FORTRAN:
      include 'PGS_SMF.f'
      include 'PGS_CSC_4.f'

      integer function pgs_csc_geotoecr(longitude,latitude,altitude,
                                        earthellipsTag,x)
      double precision  longitude
      double precision  latitude
      double precision  altitude
      character*20      earthellipstag
      double precision  posecr(3)

DESCRIPTION:
      This tool converts a geodetic latitude and longitude to ECR (Earth 
      Centered Rotating) coordinates 

INPUTS:
      Name               Description     Units    Min        Max
      ----               -----------     -----    ---        ---
      longitude          longitude       radians  -pi        pi
      latitude           latitude        radians  -pi/2      pi/2
      altitude           altitude        meters   -.1*radius N/A
      earthellipstag     earth model     N/A      N/A        N/A               
                         used

OUTPUTS:
      Name       Description         Units    Min           Max
      ----       -----------         -----    ---           ---
      posECR     ECR rectangular     meters   -100,000,000  100,000,000
                 coordinates                  (usually each component
                                              will be in range [-10,000,000,
                                              +10,000,000 m ] but function
                                              will work for Geosynchronous
                                              cases, e.g. )                 
RETURNS:
      PGS_S_SUCCESS                    success case
      PGSCSC_W_DEFAULT_EARTH_MODEL     the default earth model is used because
                                       a correct one was not specified   
      PGSCSC_W_DATA_FILE_MISSING       the data file earthfigure.dat is missing
      PGSCSC_W_SPHERICAL_BODY          using a sperical earth model
      PGSCSC_W_PROLATE_BODY            using a prolate earth model
      PGSCSC_W_LARGE_FLATTENING        issued if flattening factor is greater
                                       then 0.01  
      PGSCSC_W_INVALID_ALTITUDE        an invalid altitude was specified
      PGSCSC_E_NEG_OR_ZERO_RAD         the equatorial or polar radius is 
                                       negative or zero
      PGS_E_TOOLKIT                    something unexpected happened, execution
                                       of function ended prematurely

EXAMPLES:
C:
      PGSt_SMF_status      returnStatus
      PGSt_double          longitude
      PGSt_double          latitude
      PGSt_double          altitude
      char                 earthEllipsTag[5],
      PGSt_double          posECR[3] 
      char                 err[PGS_SMF_MAX_MNEMONIC_SIZE];  
      char                 msg[PGS_SMF_MAX_MSG_SIZE];      
     
      longitude = 0.45;
      latitude = 1.34;
      altitude = 5000.0;
      strcpy(earthEllipsTag,"WGS84"); 
       
      returnStatus = PGS_CSC_GEOtoECR(longitude,latitude,altitude,
                                       earthEllipsTag,posECR);
      if(returnStatus != PGS_S_SUCCESS)
	  {
	      PGS_SMF_GetMsg(&returnStatus,err,msg);
	      printf("\nERROR: %s",msg);
	  }
FORTRAN:
      integer              pgs_csc_geotoecr
      integer              returnstatus
      double precision     longitude
      double precision     latitude
      double precision     altitude
      character*20         earthellipstag,
      double precision     posecr(3)
      character*33 	   err
      character*241 	   msg

      longitude = 0.45
      latitude = 1.34
      altitude = 5000
      earthellipstag = 'WGS84'

      returnstatus = pgs_csc_geotoecr
                     (longitude,latitude,altitude,earthellipstag,posecr)

      if(returnstatus .ne. pgs_s_success) then
	      returnstatus = pgs_smf_getmsg(returnstatus,err,msg)
	      write(*,*) err, msg
      endif

NOTES:
      NONE
      
REQUIREMENTS:
      PGSTK - 1050, 0930

DETAILS:
      NONE

GLOBALS:
      NONE

FILES:
      earthfigure.dat

FUNCTION CALLS:
      PGS_SMF_SetStaticMsg      set error/status message
      PGS_SMF_SetDynamicMsg     set error/status message
      PGS_SMF_SetUnknownMsg     set error/status message for unknown message
      PGS_CSC_GetEarthFigure    get equatorial and polar radius
      
END_PROLOG
******************************************************************************/
#include <math.h>               /* math library  header file */
/*#include <cfortran.h>*/
#include <PGS_CSC.h>            /* Header file for Coord. Transform Tools */
#include "TS_GLOBAL.h"

PGSt_SMF_status
PGS_CSC_GEOtoECR(                     /* converts from geodetic to ECR
                                         coordinates */
    PGSt_double       longitude,      /* longitude of point on earth*/
    PGSt_double       latitude,       /* latitude of point on earth */
    PGSt_double       altitude,       /* altitude of point on earth */
    char              *earthEllipsTag,/* used for determining earth radius */
    PGSt_double       posECR[3])      /* ECR rectangular coordinates */
{
    
    PGSt_double  flatFac;        /* flattening factor - which is related to
                                    eccentricity of the ellipsoid of 
                                    revolution */
    PGSt_double  distOfPoint;    /* distance of point from Earth's axis */
    PGSt_double  recip;          /* reciprocal of the radius of curvature in a
                                    plane orthogonal to the meridian */
    PGSt_double  equatRad_A;     /* equatorial radius */
    /* PGSt_double  polarRad_C; */     /* polar radius */
    
    PGSt_SMF_status returnStatus = PGS_S_SUCCESS; /* return value of 
                                                     function and default */
      
    /* initialize return status to success */

    returnStatus = PGS_S_SUCCESS;
  
    /* default answer in case of failure */

    posECR[0] = 1.0e50;
    posECR[1] = 1.0e50;
    posECR[2] = 1.0e50;
    
    /* get equatorial and polar radius */
/*  returnStatus = PGS_CSC_GetEarthFigure(earthEllipsTag,&equatRad_A,
                                          &polarRad_C);
    switch (returnStatus)
    {
      case PGS_S_SUCCESS:
      case PGSCSC_W_DEFAULT_EARTH_MODEL:
	break;
      case PGSCSC_W_DATA_FILE_MISSING:
	return returnStatus;
      default:
	UTreportWarning(W_UT_GENERIC, "PGS_CSC_GEOtoECR()");
	PGS_SMF_SetUnknownMsg(returnStatus,"PGS_CSC_GEOtoECR()");
	return PGS_E_TOOLKIT;
    }	*/
   
    /* issue an error if the equatorial radius is negative */
/*  if (equatRad_A <= 0)
    {
	returnStatus = PGSCSC_E_NEG_OR_ZERO_RAD;
	UTreportWarning(W_UT_GENERIC, "the equatorial radius is negative or zero: PGS_CSC_GEOtoECR()");
	PGS_SMF_SetDynamicMsg(returnStatus,"the equatorial radius is negative"
 *                              "or zero",
 *			      "PGS_CSC_GEOtoECR()");
        return returnStatus;
    }	*/
    
    /* issue an error if the polar radius is negative */
/*  if (polarRad_C <= 0)
    {
	returnStatus = PGSCSC_E_NEG_OR_ZERO_RAD;
	UTreportWarning(W_UT_GENERIC, "the polar radius is negative or zero: PGS_CSC_GEOtoECR()");
	PGS_SMF_SetDynamicMsg(returnStatus,"the polar radius is negative"
 *                             "or zero",
 *                             "PGS_CSC_GEOtoECR()");
	return returnStatus;
    }	*/
    
    /* issue an error if the height is less than -0.1 * the equatorial earth
       radius */

    /* check for an invalid altitude */
    /* NOTE: ALTITUDE ALWAYS SET TO 0 IN L1 COINCIDENCE CODE WHERE THIS ROUTINE IS CALLED */
/*  if (altitude < (-0.1 * equatRad_A))
    {
	returnStatus = PGSCSC_W_INVALID_ALTITUDE;
	UTreportWarning(W_UT_GENERIC, "PGS_CSC_GEOtoECR()");
	PGS_SMF_SetStaticMsg(returnStatus,"PGS_CSC_GEOtoECR()");
        return returnStatus;
    }	*/
    
    /*  determine the flattening factor */

/*  flatFac = (equatRad_A - polarRad_C) / equatRad_A;	*/

/* NOTE: flatfac AND equatRad_A ARE NOW GOTTEN FROM TS_GLOBAL.h */
    flatFac    = FLATC;
    equatRad_A = R_EQUATOR * 1000.0;
    
    /* check for errors in the value of the flattening factor */
/*  if (flatFac < 0.0)
    {
      returnStatus = PGSCSC_W_PROLATE_BODY;
	UTreportWarning(W_UT_GENERIC, "a prolate earth model is being used:  PGS_CSC_GEOtoECR()");
        PGS_SMF_SetDynamicMsg(returnStatus,"a prolate earth model is being used *",
 *                           "PGS_CSC_GEOtoECR()");
      return returnStatus;
    }
    else if (flatFac == 0)
    {
      returnStatus = PGSCSC_W_SPHERE_BODY;
      UTreportWarning(W_UT_GENERIC, "a spherical earth model is being used:  PGS_CSC_GEOtoECR()");
      PGS_SMF_SetDynamicMsg(returnStatus,
 *                           "a spherical earth model is being used",
 *                           "PGS_CSC_GEOtoECR()");
    }
    else if (flatFac > 0.01)
    {
      returnStatus = PGSCSC_W_LARGE_FLATTENING;
      UTreportWarning(W_UT_GENERIC, "PGS_CSC_GEOtoECR()");
      PGS_SMF_SetStaticMsg(returnStatus,"PGS_CSC_GEOtoECR()");
    }	*/

    /* compute distance of point from Earth's surface */

    /* compute the reciprocal of the radius of curvature in a plane orthogonal
       to the meridian */
    recip = 1.0 / sqrt(1 - (2*flatFac - flatFac*flatFac) * sin(latitude) * 
                       sin(latitude));
    distOfPoint = ((equatRad_A * recip) + altitude) * cos(latitude);
    
    /* calculate the ECR coordinates */

    /* calculate the x coordinate */
    posECR[0] = distOfPoint * cos(longitude);

    /* calculate the y coordinate */
    posECR[1] = distOfPoint * sin(longitude);

    /* calculate the z coordinate */
    posECR[2] = ((equatRad_A * (1-flatFac) * (1-flatFac) * recip) + altitude) *
                 sin(latitude);
    
/*    if (returnStatus == PGS_S_SUCCESS)                          */
/*      UTreportWarning(W_UT_GENERIC, "PGS_CSC_GEOtoECR()");      */
/*      PGS_SMF_SetStaticMsg(returnStatus,"PGS_CSC_GEOtoECR()");  */
    return returnStatus;
}

/* definition of C function for FORTRAN binding */
/*FCALLSCFUN5(INT, PGS_CSC_GEOtoECR, PGS_CSC_GEOTOECR, \
 *		 pgs_csc_geotoecr,DOUBLE,DOUBLE,DOUBLE,STRING,DOUBLEV)
 */
