/*******************************************************************************
BEGIN_FILE_PROLOG:

FILENAME:   
   PGS_TD_TRMMtoTAI.c

DESCRIPTION:
   This file contains the function PGS_TD_TRMMtoTAI().
   This function converts TRMM spacecraft clock time in CCSDS Unsegmented Time
   Code (CUC) (with implicit P-field) format to TAI (Toolkit internal time).
 
AUTHOR:
   Guru Tej S. Khalsa /	Applied Research Corp.

HISTORY:
   09-Aug-1994  GTSK  Initial version

END_FILE_PROLOG:
*******************************************************************************/

/*******************************************************************************
BEGIN_PROLOG:

TITLE:
   Convert TRMM Clock Time to TAI Time

NAME:
   PGS_TD_TRMMtoTAI()

SYNOPSIS:
 C:
   #include <PGS_TD.h>

   PGSt_SMF_status
   PGS_TD_TRMMtoTAI(
       PGSt_scTime  scTime[8],
       PGSt_double  *secTAI93)

 FORTRAN:
      include 'PGS_TD_3.f'
      include 'PGS_SMF.f'

      integer function pgs_td_trmmtotai(sctime,sectai93)
      character*8      sctime
      double precision sectai93

DESCRIPTION:
   This function converts TRMM spacecraft clock time in CCSDS Unsegmented Time
   Code (CUC) (with implicit P-field) format to TAI (Toolkit internal time).

INPUTS:
   NAME      DESCRIPTION
   ----      -----------
   scTime    TRMM clock time in CCSDS unsegmented time code format (CUC).
	     Almost. See NOTES below for detailed description as well as
	     discussion of min and max values.

OUTPUTS:	  
   NAME      DESCRIPTION                       UNITS    MIN   MAX
   ----      -----------		       -----    ---   ---
   secTAI93  real continuous seconds since     seconds  0.0   4294967296.999999
             12AM UTC 1-1-93

RETURNS:
   PGS_S_SUCCESS               successful execution

EXAMPLES:
   N/A

NOTES:
   TIME ACRONYMS:

     TAI is:  International Atomic Time

   TRMM TIME FORMAT:

     !!!!!!!!!!!!! BEGIN WARNING: !!!!!!!!!!!!!

     This routine makes the assumption that the TRMM s/c clock is basically a
     count of continuous real seconds since 12AM UTC 1-1-1993.  This is in fact
     incorrect.  The TRMM clock is a count of the number of seconds since the
     time card (hardware) was actually powered up (but only ostensibly).  This
     clock can and may indeed actually be "jammed" (reset at arbitrary times)
     from ground control.  TRMM keeps track of (separately of course) a
     Universal Time Correlation Factor (UTCF) which is an arbitrary number
     representing real seconds that must be added to the TRMM s/c clock in order
     to derive the time.  This number accounts for several things including time
     offset correction between power-up and the epoch time, drift in the s/c
     clock that has rendered it inaccurate (accuracy requirement for TRMM s/c
     clock is nearest millisecond) AND leap-second correction which means really
     that TRMM s/c clock is rendered directly into UTC (not TAI).  The formula
     to find UTC from TRMM s/c clock time is something like:

     daysSinceEpoch = (int) (scClockTime + UTCF)/86400;
     secondsOfDay = (double) (scClockTime + UTCF) - 86400*daysSinceEpoch;

     Then decipher daysSinceEpoch into the date and secondsOfDay into the time.
     That is, the above formula yields the UTC date and time.

     The information that follows was what was known (assumed to be known as of
     6-9-94.  At this writing the exact implementation of the TRMM s/c clock has
     not been determined (by TRMM) so the code (below) stands until more info. 
     is available.

     !!!!!!!!!!!!! END WARNING !!!!!!!!!!!!!

     For TRMM, the output spacecraft clock time scTime is a 64-bit value in
     CCSDS unsegmented time code (CUC) format, which is comprised of two binary
     counter values:

     ("scTime" represents an array of 8 (unsigned) one byte elements)

     1)  A 32-bit value (the first four elements of array
         scTime each with 8 bits) containing the number of
         seconds since an epoch of 12AM UTC January 1, 1993. 
         The range of decimal values is 0-4294967295, computed as 
         256*256*256*element1 + 256*256*element2 + 256*element3 + element4. 
         The maximum decimal value of each element 1, 2, 3 and 4 is 255.
     2)  A 32-bit value (elements 5, 6, 7 and 8 of array scTime, each with 
         8 bits) containing the sub-seconds.
         The range of values is 0-4294967295 sub-seconds, computed as
         256*256*256*element3 + 256*256*element4 + 256*element5 + element6. 
         The maximum decimal value of each element 3,4,5 and 6 is 255.  The
         sub-seconds represent the fraction of the last second.  This fraction 
         is calculated as (sub-seconds)/(256^(# of sub-second elements)).

         This allows the s/c clock to represent times from 1993-01-01 to
         approximately 2129-02-07T06:26:24 (give or take a few seconds).

   TOOLKIT INTERNAL TIME (TAI):

     Toolkit internal time is the real number of continuous SI seconds since the
     epoch of UTC 12 AM 1-1-1993.  Toolkit internal time is also referred to in
     the toolkit as TAI (upon which it is based).

   REFERENCES FOR TIME:

     CCSDS 301.0-B-2 (CCSDS => Consultative Committee for Space Data Systems) 
     Astronomical Almanac, Explanatory Supplement to the Astronomical Almanac

REQUIREMENTS:  
   PGSTK-1160, PGSTK-1170

DETAILS:
   The algorithm computes the real continuous seconds since 12AM UTC 1-1-1993 
   from the input time.  This is done by converting the input time to
   integer seconds and fraction of the integer seconds.  This gives total
   seconds since s/c powerup (aka hardware time).  To this number the TRMM
   value of the Universal Time Correlation Factor (UTCF) is added to get
   total seconds since 12AM UTC 1-1-1993.  The UTCF is not available as of
   8-9-1994 and is just assumed to be zero here (highly unlikely).  It is also
   assumed to be constant here, whereas in practice it will vary.

GLOBALS:
   None

FILES:
   None

FUNCTIONS_CALLED:
   None

END_PROLOG:
*******************************************************************************/

#include <PGS_TD.h>

PGSt_SMF_status
PGS_TD_TRMMtoTAI(                 /* converts EOS PM s/c time to TAI */
    PGSt_scTime   scTime[8],      /* TRMM s/c time in CCSDS unseg. time code */
    PGSt_double   *secTAI93)      /* seconds since 12AM UTC 1-1-1993 */
{
    PGSt_uinteger seconds;        /* decimal number of s/c clock seconds */
    PGSt_uinteger subSeconds;     /* decimal number of s/c clock sub-seconds */
    
    PGSt_double   trmmUTCF=0.0;   /* Universal Time Correction Factor, this is
				     the time offset added by TRMM to get
				     seconds since 12 AM 1-1-1993 from hardware
				     time which is what is the spacecraft clock
				     actually is.  Hardware time is the seconds
				     since s/c (time card) powerup. */
    
    seconds = ((scTime[0]*256U + scTime[1])*256U + scTime[2])*256U + 
                 scTime[3];
    subSeconds = ((scTime[4]*256U + scTime[5])*256U + scTime[6])*256U + 
	         scTime[7];
    *secTAI93 = (PGSt_double) seconds + ((PGSt_double) subSeconds)/4294967296.0
                + trmmUTCF;
    
    return PGS_S_SUCCESS;          /* see WARNING in NOTES section above */
}

/* notes:   

   critical before use:
   see WARNING in NOTES section above */
