/*******************************************************************************
BEGIN_FILE_PROLOG:

FILENAME:   
   PGS_TD_TRMMtoUTC.c

DESCRIPTION:
   This file contains the function PGS_TD_TRMMtoUTC().
   This function converts TRMM spacecraft clock time in CCSDS unsegmented Time
   Code (CUC) (with implicit P-field) format to UTC in CCSDS ASCII time code A
   format.
 
AUTHOR:
   Guru Tej S. Khalsa / Applied Research Corp.

HISTORY:
   31-May-1994  GTSK  Initial version
   10-Jun-1994  GTSK  Added WARNING message to NOTES section (below) explaining
                      why this algorithm doen't actually work with the real 
		      TRMM time.

END_FILE_PROLOG:
*******************************************************************************/

/******************************************************************************
BEGIN_PROLOG:

TITLE:
   Convert TRMM Clock Time to UTC Time

NAME:
   PGS_TD_TRMMtoUTC()
 
SYNOPSIS:
 C:
   #include PGS_TD.h

   PGSt_SMF_status
   PGS_TD_TRMMtoUTC(
       unsigned char *scTime,
       char          *asciiUTC)

 FORTRAN:
      include 'PGS_TD_3.f'
      include 'PGS_SMF.f'

      integer function pgs_td_trmmtoutc(sctime,asciiutc)
      character*8      sctime
      character*27     asciiutc

DESCRIPTION:
   This function converts TRMM spacecraft clock time in CCSDS unsegmented Time
   Code (CUC) (with implicit P-field) format to UTC in CCSDS ASCII time code A
   format.
 
INPUTS:
   NAME        DESCRIPTION                UNITS        MIN            MAX
   ----        -----------                -----        ---            ---
   scTime      TRMM clock time in CCSDS         ** see NOTES below **
               unsegmented time code 
	       format (CUC).

OUTPUTS:
   NAME        DESCRIPTION                UNITS        MIN            MAX
   ----        -----------	          -----        ---            ---
   asciiUTC    UTC in CCSDS ASCII         ASCII        1993-01-01     see NOTES
               time code A format

RETURNS:
   PGS_S_SUCCESS               successful execution
   PGSTD_W_PRED_LEAPS          TAI-UTC value is predicted (not actual)
   PGSTD_E_NO_LEAP_SECS        leap seconds correction unavailable at 
	                       requested time

EXAMPLES:
   N/A

NOTES:
   TIME ACRONYMS:
     
     TAI is:  International Atomic Time
     UTC is:  Coordinated Universal Time

   TIME BOUNDARIES:

     The minimum and maximum times that can successfully be processed by this
     function depend on the file leapsec.dat which relates leap second (TAI-UTC)
     values to UTC Julian dates.  The file leapsec.dat starts at Jan 1, 1961;
     therefore an error status message will be returned if this function is
     called with a time before that date.  The file, which is updated when a new
     leap second event is announced, contains actual (definitive) and predicted
     (long term; very approximate) dates of leap second events.  The latter can
     be used only in simulations.   If an input date is outside of the range of
     dates in the file (or if the file cannot be read) the function will use
     a calculated value of TAI-UTC based on a linear fit of the data known to be
     in the table.  This value of TAI-UTC is relatively crude estimate and may
     be off by as much as 1.0 or more seconds.  Thus, when the function is used
     for dates in the future of the date and time of invocation, the user ought
     to carefully check the return status.  The status level of the return 
     status of this function will be 'W' (at least) if the TAI-UTC value used
     is a predicted value.  The status level will be 'E' if the TAI-UTC value
     is calculated (although processing will continue in this case, using the
     calculated value of TAI-UTC).

   TRMM TIME FORMAT:

     !!!!!!!!!!!!! BEGIN WARNING: !!!!!!!!!!!!!

     This routine makes the assumption that the TRMM s/c clock is basically a
     count of continuous real seconds since 12AM UTC 1-1-1993.  This is in fact
     incorrect.  The TRMM clock is a count of the number of seconds since the
     time card (hardware) was actually powered up (but only ostensibly).  This
     clock can and may indeed actually be "jammed" (reset at arbitrary times)
     from ground control.  TRMM keeps track of (separately of course) a
     Universal Time Correlation Factor (UTCF) which is an arbitrary number
     representing real seconds that must be added to the TRMM s/c clock in order
     to derive the time.  This number accounts for several things including time
     offset correction between power-up and the epoch time, drift in the s/c
     clock that has rendered it inaccurate (accuracy requirement for TRMM s/c
     clock is nearest millisecond) AND leap-second correction which means really
     that TRMM s/c clock is rendered directly into UTC (not TAI).  The formula
     to find UTC from TRMM s/c clock time is something like:

     daysSinceEpoch = (int) (scClockTime + UTCF)/86400;
     secondsOfDay = (double) (scClockTime + UTCF) - 86400*daysSinceEpoch;

     Then decipher daysSinceEpoch into the date and secondsOfDay into the time.
     That is, the above formula yields the UTC date and time.

     The information that follows was what was known (assumed to be known as of
     6-9-94.  At this writing the exact implementation of the TRMM s/c clock has
     not been determined (by TRMM) so the code (below) stands until more info. 
     is available.

     !!!!!!!!!!!!! END WARNING !!!!!!!!!!!!!

     For TRMM, the output spacecraft clock time scTime is a 64-bit value in
     CCSDS unsegmented time code (CUC) format, which is comprised of two binary
     counter values:

     ("scTime" represents an array of 8 (unsigned) one byte elements)

     1)  A 32-bit value (the first four elements of array
         scTime each with 8 bits) containing the number of
         seconds since an epoch of 12AM UTC January 1, 1993. 
         The range of decimal values is 0-4294967295, computed as 
         256*256*256*element1 + 256*256*element2 + 256*element3 + element4. 
         The maximum decimal value of each element 1, 2, 3 and 4 is 255.
     2)  A 32-bit value (elements 5, 6, 7 and 8 of array scTime, each with 
         8 bits) containing the sub-seconds.
         The range of values is 0-4294967295 sub-seconds, computed as
         256*256*256*element3 + 256*256*element4 + 256*element5 + element6. 
         The maximum decimal value of each element 3,4,5 and 6 is 255.  The
         sub-seconds represent the fraction of the last second.  This fraction 
         is calculated as (sub-seconds)/(256^(# of sub-second elements)).

         This allows the s/c clock to represent times from 1993-01-01 to
         approximately 2129-02-07T06:26:24 (give or take a few seconds).

   ASCII UTC:

     Toolkit functions that accept an ASCII time as input require the time to
     be UTC time in CCSDS ASCII Time Code A or CCSDS ASCII Time Code B format.
     Toolkit functions that return an ASCII time return the UTC time in CCSDS
     ASCII Time Code A format (see CCSDS 301.0-B-2 for details).
     (CCSDS => Consultative Committee for Space Data Systems)

      The general format is:
 
          YYYY-MM-DDThh:mm:ss.ddddddZ (Time Code A)
          YYYY-DDDThh:mm:ss.ddddddZ   (Time Code B)
 
          where:
              -,T,: are field delimiters
              Z is the (optional) time code terminator
              other fields are numbers as implied:
                Time Code A:
                   4 Year digits, 2 Month, 2 Day, 2 hour,
                   2 minute, 2 second, and up to 6 decimal
                   digits representing a fractional second
                Time Code B:
                   4 Year digits, 3 Day of year, 2 hour,
                   2 minute, 2 second, and up to 6 decimal
                   digits representing a fractional second

   REFERENCES FOR TIME:

     CCSDS 301.0-B-2 (CCSDS => Consultative Committee for Space Data Systems) 
     Astronomical Almanac, Explanatory Supplement to the Astronomical Almanac

REQUIREMENTS:  
   PGSTK-1160, PGSTK-1170

DETAILS:
   conversion method:

   The algorithm computes the real continuous seconds since 1/1/93 from the
   input time.  This number is then passed to PGS_TD_TAItoUTC() which converts
   seconds since 12AM 1/1/93 to a UTC ASCII time.

GLOBALS:
   None

FILES:
   This function calls PGS_TD_TAItoUTC() which requires the file:
   leapsec.dat

FUNCTIONS_CALLED:
        PGS_TD_TAItoUTC()

END_PROLOG:
*******************************************************************************/

#include <PGS_TD.h>
#include <string.h>

PGSt_SMF_status
PGS_TD_TRMMtoUTC(                 /* converts EOS PM s/c time to ASCII UTC */
    unsigned char *scTime,        /* TRMM s/c time in CCSDS unseg. time code */
    char          *asciiUTC)     /* equivalant UTC time in CCSDS ASCII Time
				     Code A format */
{
    PGSt_uinteger   seconds;      /* decimal number of s/c clock seconds */
    PGSt_uinteger   subSeconds;   /* decimal number of s/c clock sub-seconds */

    PGSt_double     totalSeconds; /* real number of seconds since epoch */

/************************************************************************/
/* The following was added by Yong Zhang in March 2007 to make the code */
/* machine independent.                                                 */
/************************************************************************/

    memcpy(&seconds, scTime, 4);
    memcpy(&subSeconds, scTime+4, 4);

/***********************************************************************/
/* The following was commented out by Yong Zhang in March 2007 to make */
/* the code machine independent.                                       */
/***********************************************************************/

/*
    seconds = ((scTime[0]*256U + scTime[1])*256U + scTime[2])*256U + 
	      scTime[3];

    subSeconds = ((scTime[4]*256U + scTime[5])*256U + scTime[6])*256U +
                 scTime[7];
*/

    totalSeconds = seconds + ( (double) subSeconds)/4294967296.0;
    
    return PGS_TD_TAItoUTC(totalSeconds, asciiUTC);
}

/* notes:   

   critical before use:
   see WARNING in NOTES section above */
