/*******************************************************************************
BEGIN_FILE_PROLOG:

FILENAME:   
   PGS_TD_UTCjdtoTAIjd.c

DESCRIPTION:
   This file contains the function PGS_TD_UTCjdtoTAIjd().
   The function converts UTC as a Julian date to TAI as a Julian date.
                                         
AUTHOR:
   Peter D. Noerdlinger / Applied Research Corp.
   Anubha Singhal / Applied Research Corp.
   Guru Tej S. Khalsa / Applied Research Corp.

HISTORY:
   15-Dec-1994  GTSK   Initial version--based on code by PDN and AS.
   07-Jun-1995  GTSK   Altered algorithm to check for occurrence of a negative
                       leap second event (highly unlikely, but possible).
   04-Jul-1995  GTSK   Fixed minor bug that was causing an erroneous time to be
                       returned for times very near a leap second  event IF the
		       same array was passed in for both input and output.
   12-Jul-1995  PDN    Fixed prolog.
   25-Jul-1995  GTSK   Improved efficacy and efficiency of leap second event
                       testing schemes.

END_FILE_PROLOG:
*******************************************************************************/

/*******************************************************************************
BEGIN_PROLOG:

TITLE:  
   Convert UTC Julian date to TAI Julian date

NAME:    
   PGS_TD_UTCjdtoTAIjd()

SYNOPSIS
 C:
   #include PGS_TD.h

   PGSt_SMF_status
   PGS_TD_UTCjdtoTAIjd(
          PGSt_double  jdUTC[2],
	  PGSt_boolean onLeap,
	  PGSt_double  jdTAI[2]);

FORTRAN:
      include 'PGS_SMF.f'
      include 'PGS_TD_3.f'

      integer function pgs_td_utcjdtotaijd(jdutc,onleap,jdtai)
      double precision  jdutc[2]
      integer           onleap
      double precision  jdtai[2]

DESCRIPTION:
   This tool converts UTC as a Julian date to TAI as a Julian date.

INPUTS:
   NAME      DESCRIPTION                  UNITS       MIN        MAX
   ----      -----------                  -----       ---        ---
   jdUTC     UTC as a Julian date         days        2441317.5  invocation
                                                                 (see notes)
   onLeap    set to PGS_TRUE to indicate  T/F
             that the current time is
	     occurring during a leap
	     second, otherwise set to
	     PGS_FALSE
 
OUTPUTS:
   NAME      DESCRIPTION                  UNITS       MIN        MAX
   ----      -----------                  -----       ---        ---
   jdTAI     TAI as a Julian date         days        2441317.5  invocation
                                                                 (see notes)

RETURNS:
   PGS_S_SUCCESS               successful execution
   PGSTD_E_TIME_VALUE_ERROR    error in value of input UTC time
   PGSTD_W_PRED_LEAPS          indicates that predicted leap seconds were
                               used
   PGSTD_E_NO_LEAP_SECS        assigned to errors coming back from
                               PGS_TD_Leapsec
   PGS_E_TOOLKIT               something unexpected happened, execution of
                               function terminated prematurely
EXAMPLES:
 C:
   PGSt_SMF_status      returnStatus
   PGSt_double          jdUTC[2]={2449305.5,0.0}
   PGSt_double          jdTAI[2]
   char                 err[PGS_SMF_MAX_MNEMONIC_SIZE];
   char                 msg[PGS_SMF_MAX_MSG_SIZE];  

   returnStatus = PGS_TD_UTCjdtoTAIjd(jdUTC,PGS_FALSE,jdTAI)
   if (returnStatus != PGS_S_SUCCESS)
   {
       PGS_SMF_GetMsg(&returnStatus,err,msg);
       printf("\nERROR: %s",msg);
    }

 FORTRAN:
      implicit none
      integer           pgs_td_utcjdtotaijd
      integer           returnstatus
      double precision  jdutc(2)
      double precision  jdtai(2)
      character*33      err
      character*241     msg

      jdutc(1) = 2449305.5D0
      jdutc(2) = 0.0D0
      returnstatus = pgs_td_utcjdtotaijd(jdutc,onleap,jdtai)
      if (returnstatus .ne. pgs_s_success) then
	  returnstatus = pgs_smf_getmsg(returnstatus,err,msg)
	  write(*,*) err, msg
      endif

NOTES:
   The input is UTC time as a Julian date.  There is no way to represent
   uniquely the case where a leap second is occurring (the last second of the
   day is in effect repeated).  If the input time is occurring during a leap
   second this function must be informed by setting the value of onLeap to
   PGS_TRUE.

   TIME ACRONYMS:
     
     TAI is:  International Atomic Time
     UTC is:  Coordinated Universal Time

   TIME BOUNDARIES:

     The minimum and maximum times that can successfully be processed by this
     function depend on the file leapsec.dat which relates leap second (TAI-UTC)
     values to UTC Julian dates.  The file leapsec.dat starts at Jan 1, 1961;
     therefore an error status message will be returned if this function is
     called with a time before that date.  The file, which is updated when a new
     leap second event is announced, contains actual (definitive) and predicted
     (long term; very approximate) dates of leap second events.  The latter can
     be used only in simulations.   If an input date is outside of the range of
     dates in the file (or if the file cannot be read) the function will use
     a calculated value of TAI-UTC based on a linear fit of the data known to be
     in the table.  This value of TAI-UTC is relatively crude estimate and may
     be off by as much as 1.0 or more seconds.  Thus, when the function is used
     for dates in the future of the date and time of invocation, the user ought
     to carefully check the return status.  The status level of the return 
     status of this function will be 'W' (at least) if the TAI-UTC value used
     is a predicted value.  The status level will be 'E' if the TAI-UTC value
     is calculated (although processing will continue in this case, using the
     calculated value of TAI-UTC).

   JULIAN DATES:

     Format:

       Toolkit Julian dates are kept as an array of two real (high precision)
       numbers (C: PGSt_double, FORTRAN: DOUBLE PRECISION).  The first element
       of the array should be the half integer Julian day (e.g. N.5 where N is a
       Julian day number).  The second element of the array should be a real
       number greater than or equal to zero AND less than one (1.0) representing
       the time of the current day (as a fraction of that (86400 second) day.
       This format allows relatively simple translation to calendar days (since
       the Julian days begin at noon of the corresponding calendar day).  Users
       of the Toolkit are encouraged to adhere to this format to maintain high
       accuracy (one number to track significant digits to the left of the
       decimal and one number to track significant digits to the right of the
       decimal).  Toolkit functions that do NOT require a Julian type date as an
       input and return a Julian date will return the Julian date in the above
       mentioned format.  Toolkit functions that require a Julian date as an
       input and do NOT return a Julian date will first convert the input date
       (internal) to the above format.  Toolkit functions that have a Julian
       date as both an input and an output will assume the input is in the above
       described format but will not check and the format of the output may not
       be what is expected if any other format is used for the input.

     Meaning:

       Toolkit "Julian dates" are all derived from UTC Julian Dates.  A Julian
       date in any other time stream (e.g. TAI, TDT, UT1, etc.) is the UTC
       Julian date plus the known difference of the other stream from UTC
       (differences range in magnitude from 0 seconds to about a minute).

       Examples:

         In the following examples, all Julian Dates are expressed in Toolkit
         standard form as two double precision numbers. For display here, the
         two members of the array are enclosed in braces {} and separated by a
         comma.

         A) UTC to TAI Julian dates conversion

         The Toolkit UTC Julian date for 1994-02-01T12:00:00 is: 
         {2449384.50, 0.5}.
         TAI-UTC at 1994-02-01T12:00:00 is 28 seconds (.00032407407407 days). 
         The Toolkit TAI Julian date for 1994-02-01T12:00:00 is:
         {2449384.50, 0.5 + .00032407407407} = {2449384.50, 0.50032407407407}

         Note that the Julian day numbers in UTC and the target time stream may
         be different by + or - 1 for times near midnight:

         B) UTC to UT1 Julian dates conversion

         The Toolkit UTC Julian date for 1994-04-10T00:00:00 is: 
         {2449452.50, 0.0}.
         UT1-UTC at 1994-04-10T00:00:00 is -.04296 seconds 
         (-0.00000049722221 days).  The Toolkit UT1 Julian date for
         1994-04-10T00:00:00 is:
         {2449452.50, 0.0 - 0.0000004972222} = 
         {2449452.50, -0.0000004972222} =
         {2449451.50, 0.9999995027778}

   REFERENCES FOR TIME:

     CCSDS 301.0-B-2 (CCSDS => Consultative Committee for Space Data Systems) 
     Astronomical Almanac, Explanatory Supplement to the Astronomical Almanac

REQUIREMENTS:
   PGSTK - 1160, 1170, 1210

DETAILS:
      See "Theoretical Basis of the SDP Toolkit Geolocation Package for the
      ECS Project", Document 445-TP-002-002, May 1995, by P. Noerdlinger, 
      for more information on the algorithm.

GLOBALS:
   None

FILES:
   PGS_TD_LeapSec requires leapsec.dat
         	  
FUNCTIONS_CALLED:
   PGS_TD_LeapSec          -    get leap second value
   PGS_SMF_SetStaticMsg    -    set error/status message
   PGS_SMF_SetDynamicMsg   -    set error/status message
   PGS_SMF_SetUnknownMsg   -    set unknown error/status message

END_PROLOG:
*******************************************************************************/

#include <string.h>
#include <PGS_TD.h>               /*  Header file for Time and Date Tools */
#include "TS_GLOBAL.h"
       
/* name of this function */

#define FUNCTION_NAME   "PGS_TD_UTCjdtoTAIjd()"

/* constants */

#define SECONDSperDAY     86400.0 /* number of seconds in a day */

PGSt_SMF_status
PGS_TD_UTCjdtoTAIjd(              /* converts UTC as a Julian date to TAI as a
				     Julian date  */
    PGSt_double  jdUTC[2],        /* UTC as a Julian date */
    PGSt_boolean onLeap,          /* set to true if UTC time is occurring during
				     a positive leap second */
    PGSt_double  jdTAI[2])        /* TAI as a Julian date */
{
    PGSt_double  jdTemp[2];       /* temporary Julian date array */
    PGSt_double  leapSecs;        /* leap seconds value from tables read 
                                     by PGS_TD_LeapSec()  */
    PGSt_double  newleapSecs;     /* leap seconds value from tables read
                                     by second call to PGS_TD_LeapSec()  */
    PGSt_double  lastChangeJD;    /* Julian day number on which leap second
				     returned by PGS_TD_LeapSec() became (or is
				     predicted to become) effective */
    PGSt_double  newLastChangeJD; /* Julian day number on which leap second
				     returned by PGS_TD_LeapSec() became (or is
				     predicted to become) effective (used in
				     second call to PGS_TD_LeapSec()) */
    PGSt_double  nextChangeJD;    /* Julian day number on which next leap second
				     value became (or is predicted to become)
				     effective */
    PGSt_integer currentLeap=0;   /* current leap second value (0 or 1) as
				     determined by onLeap */
    char         leapStatus[10];  /* flag indicating actual or predicted leap
                                     seconds - for user-defined diagnostics
                                     and latency checks */
    PGSt_SMF_status returnStatus; /* default return value of function */
    PGSt_SMF_status returnStatus1;/* status/error return of SetStaticMsg */
    
    /* Start Executable Code */
    
    /* initialize returnStatus to indicate success */

    returnStatus = PGS_S_SUCCESS;
    
    /* Access Leap Seconds file; obtain current leap second value */

    returnStatus = PGS_TD_LeapSec(jdUTC, &leapSecs, &lastChangeJD, 
				  &nextChangeJD, leapStatus);
        
    /* check for error, reassign some of the errors coming back from 
       PGS_TD_LeapSec() since they are really more specific than
       required, set dynamic message to pass on specifics */
    
      switch (returnStatus)
      {
	case PGS_S_SUCCESS:


	  /* issue a warning if the difference of TAI and UTC is predicted */
	  
	  if (strcmp(leapStatus,"PREDICTED") == 0)
	  {
	      returnStatus = PGSTD_W_PRED_LEAPS;
/*	      PGS_SMF_SetStaticMsg(returnStatus,FUNCTION_NAME);*/
	  }

	  /* If onLeap is set to true, then call PGS_TD_Leapsec with
	     the next julian date value. If the last change day is the same as
	     the last change day from the previous call, then a leap second
	     boundary has not been crossed and this is an invalid date for a
	     leap second. */

          if (onLeap == PGS_TRUE)
          {
	      /* set currentLeap to 1 */

	      currentLeap = 1;
	      
	      jdTemp[0] = jdUTC[0];
	      jdTemp[1] = jdUTC[1] + 1.0/SECONDSperDAY;
	  
	      returnStatus1 = PGS_TD_LeapSec(jdTemp,&newleapSecs,
					     &newLastChangeJD,&nextChangeJD,
					     leapStatus);

	      /* if the return status is not PGS_S_SUCCESS, we have most likely
		 fallen off the end of the table, don't harass users about leap
		 seconds if this is the case since the values are unknown at
		 that point anyway */

	      if (returnStatus1 != PGS_S_SUCCESS)
		  break;

              if ((jdTemp[1] > (1.0 + (newleapSecs-leapSecs)/SECONDSperDAY)) ||
		  (newLastChangeJD == lastChangeJD))
	      {
	           returnStatus = PGSTD_E_TIME_VALUE_ERROR;
/*	           PGS_SMF_SetDynamicMsg(returnStatus,"error in UTC Julian "
*					 "date: invalid date for leap second",
*                                        FUNCTION_NAME);
*/
              }

          }

	  /* If the input date is very near to the next date of change of leap
	     second value then check for the occurrence of a negative leap
	     second (very unlikely, but possible). */

	  else if ((jdUTC[0] + 1.0) == nextChangeJD)
	  {
	      /* if the day fraction is not very near midnight then there is no
		 need to check for a negative leap second occurrence, so break
		 here in that case (actually there is no need to check for the
		 occurrence of a leap second unless the day fraction indicates a
		 time of 23:59:59 or greater, but let's not push machine
		 accuracy by testing right at the boundary only) */

	      if (jdUTC[1] < .999)
		  break;
	      
	      jdTemp[0] = jdUTC[0];
	      jdTemp[1] = jdUTC[1] + 1.0/SECONDSperDAY;
	  
	      returnStatus1 = PGS_TD_LeapSec(jdTemp,&newleapSecs,&lastChangeJD,
					     &nextChangeJD,leapStatus);

	      /* if the return status is not PGS_S_SUCCESS, we have most likely
		 fallen off the end of the table, don't harass users about leap
		 seconds if this is the case since the values are unknown at
		 that point anyway */

	      if (returnStatus1 != PGS_S_SUCCESS)
		  break;

	      /* If (newleapSecs < leapSecs) then the next leap second event
		 is a negative leap second and the quantity 
		 (1.0 - (leapSecs - newleapSecs)/SECONDSperDAY) will be the
		 largest value that the current day fraction can be.  If the
		 current day fraction is greater than this value then it is
		 occurring during a UTC time which has been precluded by the
		 occurrence of the negative leap second event. */

	      if (jdUTC[1] >= (1.0 - (leapSecs - newleapSecs)/SECONDSperDAY))
	      {
	           returnStatus = PGSTD_E_TIME_VALUE_ERROR;
/*	           PGS_SMF_SetDynamicMsg(returnStatus,"error in UTC Julian "
*					 "date: a negative leap second event "
*					 "precludes input date",FUNCTION_NAME);
*/
              }
	  }
	  break;
	case PGSTD_W_JD_OUT_OF_RANGE:
	  returnStatus = PGSTD_E_NO_LEAP_SECS;
/*	  PGS_SMF_SetDynamicMsg(
*		      returnStatus,
*		      "input Julian day out of range for tabulated corrections"
*		      " - approximate value used",
*		      FUNCTION_NAME);
*/
	  break;
	case PGSTD_W_DATA_FILE_MISSING:
	  returnStatus = PGSTD_E_NO_LEAP_SECS;
/*	  PGS_SMF_SetDynamicMsg(
*		      returnStatus,
*		     "unable to find or open leap second correction file:"
*                     " leapsec.dat - an approximate value was used",
*		     FUNCTION_NAME);
*/
	  break;
	default:
/*	  PGS_SMF_SetUnknownMsg(returnStatus,FUNCTION_NAME);*/
	  return PGS_E_TOOLKIT;
      }

    /* calculate the TAI julian date:
       the Julian date is the Julian day and the fraction of the day */
    
    jdTAI[0] = jdUTC[0];
    jdTAI[1] = jdUTC[1] + ((leapSecs + currentLeap) / SECONDSperDAY);
    
    /* check to see if the day fraction is greater than or equal one - if so, 
       then add a day to the Julian day number and subtract one from the day 
       fraction so that the day fraction is between 0.0 and 1.0 */ 

    if (jdTAI[1] >= 1.0)
    {
	jdTAI[0] +=  1.0;
	jdTAI[1] -=  1.0;
    }
    
    /* return to calling function */

/*    if (returnStatus == PGS_S_SUCCESS)
 *       PGS_SMF_SetStaticMsg(returnStatus,FUNCTION_NAME);
*/
    return returnStatus;
}
