/*******************************************************************************
BEGIN_FILE_PROLOG:

FILENAME:   
   PGS_TD_UTCtoTRMM.c

DESCRIPTION:
   This file contains the function PGS_TD_UTCtoTRMM().
   This function converts UTC in CCSDS ASCII time code A (or B) format
   to TRMM spacecraft clock time in CCSDS Unsegmented Time Code (CUC) (with
   implicit P-field) format .
 
AUTHOR:
   Guru Tej S. Khalsa / Applied Research Corp.

HISTORY:
   09-Jun-1994  GTSK  Initial version
   10-Jun-1994  GTSK  Added WARNING message to NOTES section (below)
	              explaining why this algorithm doesn't actually
		      work with the real TRMM time.

END_FILE_PROLOG:
*******************************************************************************/

/*******************************************************************************
BEGIN_PROLOG:

TITLE:
   Convert UTC Time to TRMM Clock Time

NAME:
   PGS_TD_UTCtoTRMM()
 
SYNOPSIS:
   #include <PGS_TD.h>

   PGSt_SMF_status
   PGS_TD_UTCtoTRMM(
      char         asciiUTC[28],
      PGSt_scTime  scTime[8])

DESCRIPTION:
   This function converts UTC in CCSDS ASCII time code A (or B) format to TRMM
   spacecraft (s/c) clock time in CCSDS Unsegmented Time Code (CUC) (with
   implicit P-field) format.
 
INPUTS:
   NAME         DESCRIPTION                UNITS     MIN         MAX
   ----         -----------                -----     ---         ---
   asciiUTC     UTC time in CCSDS ASCII     N/A      1993-001    ** see NOTES **
                Time Code (A or B format)

OUTPUTS:
   NAME         DESCRIPTION                UNITS     MIN         MAX
   ----         -----------                -----     ---         ---
   scTime       TRMM clock time in CCSDS     ** see NOTES below **
	        unsegmented time code 
		format (CUC).

RETURNS:
   PGS_S_SUCCESS               successful execution
   PGSTD_W_PRED_LEAPS          TAI-UTC value is predicted (not actual)
   PGSTD_E_NO_LEAP_SECS        leap seconds correction unavailable at 
                               requested time
   PGSTD_E_TIME_FMT_ERROR      error in format of ascii UTC time
   PGSTD_E_TIME_VALUE_ERROR    error in value of the ascii UTC time
   PGSTD_E_DATE_OUT_OF_RANGE   input UTC time is not within the possible values
                               of the s/c clock
   PGS_E_TOOLKIT               an unexpected error occurred

EXAMPLES:
   N/A

NOTES:
   UTC is:  Coordinated Universal Time

   TIME BOUNDARIES:

     The minimum and maximum times that can successfully be processed by this
     function depend on the file leapsec.dat which relates leap second (TAI-UTC)
     values to UTC Julian dates.  The file leapsec.dat starts at Jan 1, 1961;
     therefore an error status message will be returned if this function is
     called with a time before that date.  The file, which is updated when a new
     leap second event is announced, contains actual (definitive) and predicted
     (long term; very approximate) dates of leap second events.  The latter can
     be used only in simulations.   If an input date is outside of the range of
     dates in the file (or if the file cannot be read) the function will use
     a calculated value of TAI-UTC based on a linear fit of the data known to be
     in the table.  This value of TAI-UTC is relatively crude estimate and may
     be off by as much as 1.0 or more seconds.  Thus, when the function is used
     for dates in the future of the date and time of invocation, the user ought
     to carefully check the return status.  The status level of the return 
     status of this function will be 'W' (at least) if the TAI-UTC value used
     is a predicted value.  The status level will be 'E' if the TAI-UTC value
     is calculated (although processing will continue in this case, using the
     calculated value of TAI-UTC).

   ASCII UTC:

     Toolkit functions that accept an ASCII time as input require the time to
     be UTC time in CCSDS ASCII Time Code A or CCSDS ASCII Time Code B format.
     Toolkit functions that return an ASCII time return the UTC time in CCSDS
     ASCII Time Code A format (see CCSDS 301.0-B-2 for details).
     (CCSDS => Consultative Committee for Space Data Systems)

      The general format is:
 
          YYYY-MM-DDThh:mm:ss.ddddddZ (Time Code A)
          YYYY-DDDThh:mm:ss.ddddddZ   (Time Code B)
 
          where:
              -,T,: are field delimiters
              Z is the (optional) time code terminator
              other fields are numbers as implied:
                Time Code A:
                   4 Year digits, 2 Month, 2 Day, 2 hour,
                   2 minute, 2 second, and up to 6 decimal
                   digits representing a fractional second
                Time Code B:
                   4 Year digits, 3 Day of year, 2 hour,
                   2 minute, 2 second, and up to 6 decimal
                   digits representing a fractional second

   !!!!!!!!!!!!! BEGIN WARNING: !!!!!!!!!!!!!
   
   The following information and the rest of this routine are all incorrect.
   This routine makes the assumption that the TRMM s/c clock is basically a
   count of continuous real seconds since 12AM UTC 1-1-1993.  This is in fact
   incorrect.  The TRMM clock is a count of the number of seconds since the time
   card (hardware) was actually powered up (but only ostensibly).  This clock
   can and may indeed actually be "jammed" (reset at arbitrary times) from
   ground control.  TRMM keeps track of (separately of course) a Universal Time
   Correlation Factor (UTCF) which is an arbitrary number representing real
   seconds that must be added to the TRMM s/c clock in order to derive the time.
   This number accounts for several things including time offset correction
   between power-up and the epoch time, drift in the s/c clock that has rendered
   it inaccurate (accuracy requirement for TRMM s/c clock is nearest
   millisecond) AND leap-second correction which means really that TRMM s/c
   clock is rendered directly into UTC (not TAI).  The formula to find UTC from
   TRMM s/c clock time is something like:
   
   daysSinceEpoch = (int) (scClockTime + UTCF)/86400;
   secondsOfDay = (double) (scClockTime + UTCF) - 86400*daysSinceEpoch;
   
   Then decipher daysSinceEpoch into the date and secondsOfDay into the time.
   That is, the above formula yields the UTC date and time.
   
   The information that follows was what was known (assumed to be known as of
   6-9-94.  At this writing the exact implementation of the TRMM s/c clock has
   not been determined (by TRMM) so the code (below) stands until more info. is
   available.
   
   !!!!!!!!!!!!! END WARNING !!!!!!!!!!!!!
   
   TRMM TIME FORMAT:

     For TRMM, the output spacecraft clock time scTime is a 64-bit value in
     CCSDS unsegmented time code (CUC) format, which is comprised of two binary
     counter values:
                   
     1)  A 32-bit value (the first four elements of array
         scTime each with 8 bits) containing the number of
         seconds since an epoch of 12AM UTC January 1, 1993. 
         The range of decimal values is 0-4294967295, computed as 
         256*256*256*element1 + 256*256*element2 + 256*element3
         + element4. The maximum decimal value of each 
         element 1, 2, 3 and 4 is 255.
     2)  A 32-bit value (elements 5, 6, 7 and 8 of array
         scTime, each with 8 bits) containing the sub-seconds.
         The range of values is 0-4294967295 sub-seconds,
         computed as 256*256*256*element3 + 256*256*element4 + 
         256*element5 + element6. The maximum decimal value of
         each element 3,4,5 and 6 is 255.  The sub-seconds represent 
         the fraction of the last second.  This fraction is calculated
         as (sub-seconds)/(256^(# of sub-second elements)).
         
         This allows the s/c clock to represent times from 1993-01-01 to
         approximately 2129-02-07T06:26:24 (give or take a few seconds).

   REFERENCES FOR TIME:

     CCSDS 301.0-B-2 (CCSDS => Consultative Committee for Space Data Systems) 
     Astronomical Almanac, Explanatory Supplement to the Astronomical Almanac

REQUIREMENTS:  
   PGSTK-1160, PGSTK-1170

DETAILS:
   The input time is passed to PGS_TD_UTCtoTAI() which converts
   the ASCII UTC time to real continuous seconds since 12AM UTC 1-1-93.
   The time is then checked to make sure it falls within the allowable
   range of the EOS PM spacecraft clock. These seconds are then 
   separated into integer and fractional components and decomposed into
   the appropriate octets of the TRMM spacecraft clock time (see NOTES above).

GLOBALS:
   None

FILES:
   None

FUNCTIONS_CALLED:
   PGS_TD_UTCtoTAI()
   PGS_SMF_SetDynamicMsg()
   PGS_SMF_SetUnknownMsg()

END_PROLOG:
*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <PGS_TD.h>
#include "TS_GLOBAL.h"

/* name of this function */

#define FUNCTION_NAME "PGS_TD_UTCtoTRMM()"

PGSt_SMF_status
PGS_TD_UTCtoTRMM(           /* convert UTC time to TRMM s/c clock time */
    char         *asciiUTC, /* UTC time (in ASCII) (input) */
    PGSt_scTime  scTime[8]) /* s/c clock time (in CUC) (output) */
{
    PGSt_uinteger  secondsSinceEpoch;         /* seconds since 12AM 1-1-58 */
    PGSt_uinteger  subSecondsSinceEpoch;      /* sub seconds of last second */
    PGSt_double    secTAI93;                  /* TAI time equivalent of 
						 input UTC time */

    char           msg[PGS_SMF_MAX_MSG_SIZE]; /* temporary string holder */

    PGSt_SMF_status   returnStatus;           /* return status of PGS function
						 calls */
    
    /*******************
     * BEGIN EXECUTION *
     *******************/

    /* initialize returnStatus to indicate success */

    returnStatus = PGS_S_SUCCESS;
    
    /* Call PGS_TD_UTCtoTAI to convert input UTC time to TAI time in
       seconds since 12AM UTC 1-1-93. */

    returnStatus = PGS_TD_UTCtoTAI(asciiUTC,&secTAI93);
    switch (returnStatus)
    {
      case PGS_S_SUCCESS:
      case PGSTD_W_PRED_LEAPS:
      case PGSTD_E_NO_LEAP_SECS:
	break;
      case PGSTD_E_TIME_FMT_ERROR:
      case PGSTD_E_TIME_VALUE_ERROR:
      case PGS_E_TOOLKIT:
	return returnStatus;	
      default:
/*	PGS_SMF_SetUnknownMsg(returnStatus,FUNCTION_NAME);*/
	return PGS_E_TOOLKIT;
    }
    
    /* Check to ensure that the input time is in the range of the s/c clock.
       The lowest value of the spacecraft clock is zero so secTAI93 should not
       be less than 0.0 (which condition would yield an overall 
       negative time since epoch which cannot be represented by the s/c
       clock).  There are four octets of course time (seconds) allowed by
       the s/c clock which yields a max possible value of 2^32-1 so the integer
       seconds since epoch had better be less than 2^32 (4294967296.0). */
    
    if (secTAI93 < 0.0 || secTAI93 >= 4294967296.0)
    {
	sprintf(msg,"%s%s%s","The date portion of the input UTC time ",
		             "string is out of range for TRMM ",
	                     "(12-31-92 < date < 2-8-2129)");
/*	PGS_SMF_SetDynamicMsg(PGSTD_E_DATE_OUT_OF_RANGE,msg,FUNCTION_NAME);*/
	return PGSTD_E_DATE_OUT_OF_RANGE;
    }

    /* Determine the integer seconds since epoch by casting the total (real)
       seconds since epoch as an unsigned integer (this will truncate the
       real seconds (not round) which is the desired result).  The sub-seconds
       field is four octets so the sub-seconds are the fraction of 2^32
       (4294967296) determined by multiplying the fraction part of the real
       second by 2^32 and casting to an unsigned integer (the .5 is added to
       achieve the effect of rounding which is desired in this case). */
    
    secondsSinceEpoch = (PGSt_uinteger) secTAI93;
    subSecondsSinceEpoch = (PGSt_uinteger) ((secTAI93 - secondsSinceEpoch)*
					    4294967296. + .5);

    /* Decompose the times to the s/c clock octets.  See NOTES above */

/************************************************************************/
/* The following was added by Yong Zhang in March 2007 to make the code */
/* machine independent.                                                 */
/************************************************************************/

    memcpy(scTime, &secondsSinceEpoch, 4);
    memcpy(scTime+4, &subSecondsSinceEpoch, 4);

/***********************************************************************/
/* The following was commented out by Yong Zhang in March 2007 to make */
/* the code machine independent.                                       */
/***********************************************************************/

/*
    scTime[0] = (PGSt_scTime) (secondsSinceEpoch/16777216U);
    scTime[1] = (PGSt_scTime) ((secondsSinceEpoch/65536U)%256U);
    scTime[2] = (PGSt_scTime) ((secondsSinceEpoch/256U)%256U);
    scTime[3] = (PGSt_scTime) (secondsSinceEpoch%256U);

    scTime[4] = (PGSt_scTime) (subSecondsSinceEpoch/16777216U);
    scTime[5] = (PGSt_scTime) ((subSecondsSinceEpoch/65536U)%256U);
    scTime[6] = (PGSt_scTime) ((subSecondsSinceEpoch/256U)%256U);
    scTime[7] = (PGSt_scTime) (subSecondsSinceEpoch%256U);
*/  

    return returnStatus; /* see WARNING in NOTES section above */

}

/* notes:

   critical before use:
   see WARNING in NOTES section above */
