/* CVS Comment Section
 * 
 * $Log: SPAgetBVal.c,v $
 * Revision 1.1.1.1  2011/12/30 02:47:49  hensley
 * First import
 *
 * 
 * Checked into sturmmitte CVS as "spacomlin" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * Revision 1.3  2011/07/01 22:21:19  hensley
 * Changed TSDIS to PPS
 * 
 * Revision 1.2  2008/08/20 21:33:00  hensley
 * Removed stdio.h and math.h includes
 * 
 * Revision 1.1.1.1  2008/08/19 21:17:02  hensley
 * First Import
 * 
 * Checked into sturmmitte CVS as "spaweb" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * Revision 1.1.1.1  2008/06/18 23:06:08  hensley
 * Baseline PPS-- Code
 * 
 * Linux CVS Comments When Imported as "spa" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * SGI CVS Comments vvv
 * 
 * Revision 1.4  2005/12/13 18:44:51  hensley
 * Fixed the CVS Header Section
 * 
 * Revision 1.3  1997/03/26 08:13:13  bilanow
 * Sign of orbit precession correction changed.
 * 
 * Revision 1.2  1996/11/20  23:54:04  bilanow
 * Swath model constants added to calling sequence, and various
 * constants replaced by #define values from SPA.h file.
 * 
 * Revision 1.1  1996/11/14  18:53:04  dbazell
 * moved file from spat/ to spa/
 * 
 * Revision 1.1  1996/09/20  19:32:33  bilanow
 * Added to CVS first time.
 * 
 **/

/***********************************************************************
@ Function Statement:
@ -------------------
@   void SPAgetBVal(float theta, float phi, float gamma, 
@                   float orbit_period, float orbit_precession,
@                   float earth_rate, float *lon)
@
@  Description:
@  ------------
@              This function evaluates the swath boundary function as
@              computed according to a spherical trigonometry model.
@              This model is described in a memo dated October 23, 1995,
@              from Steve Bilanow regarding:  An Efficient Search 
@              Algorithm for Coincidence of Swath Granules with 
@              Latitude/Longitude Spatial Coverage (Section 2).
@              As described in this memo, upper and lower boundary
@              functions are needed to describe the longitude of the 
@              swath edge relative to a reference (the Longitude of 
@              Maximum Latitude), as a function of the latitude of the 
@              swath edge.  
@
@              The spherical trigonometry model is configured in an
@              inertial coordinate frame with the orbit plane defining
@              the swath, and then corrections are made for Earth
@              rotation and orbit precession to get longitudes for the
@              swath edges.  The sides of the triangle are the arcs
@              theta, phi, and gamma.  For this model, the input theta
@              is a swath model parameter associated with the orbit
@              inclination.  The input phi provides a swath model
@              parameter controlled by the swath half-width and defines
@              either the upper or lower boundary function for the
@              swath.  The input gamma is associated with the latitude
@              for which the relative longitude, *lon, of the swath
@              functions needs to be computed.  The value of gamma is
@              the co-latitude of a swath edge point, or the complement
@              of the geocentric latitude.
@
@              The input angles for this computation are shown in a 
@              diagram in Figure 4 of the October 23 memo noted above.
@              A very rough schematic of the triangle is shown below
@              (shown somewhat better with a non-proportional font):
@ 
@                    Earth Pole
@                        P  
@                       /  \  theta
@                      /    \  
@            gamma    /      N  Orbit Normal
@                    /     /  
@                   /   /
@                  //     phi
@                 E  
@           swath edge point
@
@              The following description summarizes the relationships:
@              We consider the swath defined by a band in inertial
@              coordinates and compute relationships along the
@              band/swath edge according to a spherical triangle defined
@              by the points:
@
@                 P --  The Earth's spin axis at the North pole
@
@                 N --  The orbit normal direction
@
@                 E --  A point on the edge of the swath
@
@              The following arc angles form the edges of this triangle:
@
@                 theta -  The arc from P to N,
@
@                 phi   -  The arc from N to E
@
@                 gamma -  The arc from P to E
@
@
@               The following internal angles of this triangle are
@               computed in this subroutine using the half angle sine
@               formula for spherical triangles.  The capital letters
@               are used here to denote distinct variables according to
@               the spherical trig convention of labelling opposite
@               angles to the sides as noted in small greek letters.
@               (Remember PHI is different than phi in the code here as
@                well.)
@
@                 PHI   -  The rotation about P from E to N
@
@                 GAMMA -  The rotation angle about N from P to E
@
@               Note that the swath boundary function is desired stay
@               flat at zero for all low values of gamma, and stay flat
@               at a fixed maximum above certain values for gamma.  This
@               is implemented in the code by trapping conditions on the
@               value of variables noted as T1 and T2 before taking an
@               arcsin of them.  Currently no flag is returned about the
@               status of the trapping of this condition, since it is
@               not needed.
@
@               The PDL included here was developed in November 1995
@               based on the prototype code which was developed in
@               October 1995.
@
@ Input:
@ ------
@           theta  -  The arc in radians from the Pole to orbit normal
@                     (nominally 35 degrees for the TRMM orbit).
@
@           phi    -  The arc in radians from orbit normal to a swath
@                     edge point, given by (pi over 2) plus or minus the
@                     swath half-width depending whether the upper or
@                     lower swath boundary is needed.
@
@           gamma  -  The colatitude ((pi over 2) minus the latitude
@                     in radians) of the swath edge point, in radians.
@                     This is assumed to be based on a geocentric
@                     latitude for this swath mode.
@
@
@           orbit_period      -   Swath model parameter for the orbit
@                                 period (seconds)
@
@           orbit_precession  -   Swath model parameter for the orbit
@                                 precession rate (degrees/orbit)
@
@           earth_rate        -   Swath model parameter for the earth
@                                 rotation rate (degrees/second)
@
@ Output:
@ -------
@           lon    -  The relative longitude (relative to the 
@                     longitude-of-max-latitude) of the selected swath
@                     edge point defined by the input parameters.  This
@                     output is put in degrees, because those are the
@                     convenient units for use by the PPS calling
@                     program, SPAspatialBound.
@
@ Return:   None  
@ -------
@
@ Called By:  SPAspatialBound
@ ----------
@
@ Routines called:  None
@ ----------------
@
@
@   Date     Programmer Name    Change Description 
@   ----     ---------------    ---------------------------------------
@ 96-09-06   Steve Bilanow      Prototype Enhacement to Initial Build
@                               Version 
@
@ 96-11-20   Steve Bilanow      Swath model constants added to calling
@                               sequence (replacing #define constants),
@                               and constants 0.0,1.0,2.0,180.0, and
@                               360.0 replaced by #define values from
@                               SPA.h
@
@ 97-03-25   Steve Bilanow      Sign of application of orbit precession
@                               correction changed (nominal value is 
@                               negative to give proper correction)
@
@***********************************************************************
@ Algorithm (PDL)
@ ---------------
@
@ Compute sigma 
@                   (as defined in 10-23 memo, page 5,
@                    which is = (theta+phi+gamma)/2) )
@
@ Evaluate the arguments of the inverse trig functions, and
@ set appropriately to clip at upper and lower bounds as needed:
@ -------------------------------------------------------------
@
@  Compute T1 as the right side of half-angle formula for rotation angle
@  PHI - October 23 Memo, top of page 5:
@ 
@  T1 = sin(sigma-theta)*sin(sigma-gamma)/(sin(theta)*sin(gamma) )
@ 
@  IF T1 is greater than 1.0
@  THEN 
@      set T1 = 1.0
@  ELSE
@     IF T1 is less than 0.0
@     THEN 
@        set T1 = 0.0
@     ENDIF
@  ENDIF
@ 
@  Compute T2 as the right side of half-angle formula for rotation angle
@  GAMMA - October 23 Memo, middle of page 5:
@ 
@  T2 = sin(sigma-theta)*sin(sigma-phi)/(sin(theta)*sin(phi) )
@ 
@  IF T2 is greater than 1.0
@  THEN 
@      set T2 = 1.0
@  ELSE
@     IF T2 is less than 0.0
@     THEN 
@        set T2 = 0.0
@     ENDIF
@  ENDIF
@ 
@ Evaluate inverse trig functions:
@ -------------------------------
@ 
@ Compute PHI as twice the arcsine of the square root of T1
@ 
@ Compute GAMMA as twice the arcsine of the square root of T2
@ 
@ Compute corrections for Earth rotation and orbit precession:
@ -----------------------------------------------------------
@ 
@ Compute the approximate orbit time to progress the rotation angle
@ GAMMA
@ 
@ Compute the Earth rotation in this time
@ 
@ Compute a correction for orbit precession in this time
@ 
@ Get relative longitude result and return:
@ ----------------------------------------
@ 
@ Compute the relative longitude from PHI corrected for Earth rotation 
@ and orbit precession 
@ 
@ return to calling program
@ 
@**********************************************************************/

#include "SPAproto.h"

        /****************************************************
        *   Notes that key constants for the swath model    *
        *   are defined in the SPA.h file.                  *
        ****************************************************/


void SPAgetBVal(float theta, float phi, float gamma, 
                float orbit_period, float orbit_precession,
                float earth_rate, float *lon)
{

                   /*************************/
                   /*************************/
                   /* Variable Declarations */
                   /*************************/
                   /*************************/

   float sigma;         /*  term in spherical trigonometry formula    */
                        /*  ( sum of sides' arcs divided by two )     */
   float PHI;           /*  rotation angle PHI defined above          */
   float GAMMA;         /*  rotation angle GAMMA defined above        */
   float argPHI;        /*  argument of function for evaluating PHI   */
   float argGAMMA;      /*  argument of function for evaluating GAMMA */
   float tr;            /*  Time for progression in orbit by GAMMA    */
   float delta;         /*  Earth rotation during orbit arc GAMMA     */
   float epsilon;       /*  Correction for orbit precession           */

             /**********************************************************
             *   Compute sigma for the spherical trig formuala;        *
             *   this is the sum of the interior angles divided by two *
             *  (as defined in 10-23 memo, page 5,                     *
             *   i.e.  = (theta+phi+gamma)/2) )                        *
             **********************************************************/

   sigma = (theta + gamma + phi)/SPA_TWO_POINT_ZERO;

                /******************************/
                /******************************/
                /* Evaluate the arguments of  */
                /* inverse trig functions and */
                /* set appropriately to clip  */
                /* these functions at upper   */
                /* and lower bounds as needed */
                /******************************/
                /******************************/

       /***************************************************************/
       /* Compute argPHI (T1) as right side of half-angle formula for */
       /* rotation angle PHI                                          */
       /* = sin(sigma-theta)*sin(sigma-gamma)/(sin(theta)*sin(gamma)) */
       /* (October 23 Memo, top of page 5) & limit between 0.0 & 1.0  */
       /***************************************************************/

   argPHI =   sin( sigma - theta ) * sin( sigma - gamma ) /
            ( sin(         theta ) * sin(         gamma ) );

   if (argPHI > SPA_ONE_POINT_ZERO)
   {
       argPHI = SPA_ONE_POINT_ZERO;
   }
   else
   {
       if (argPHI < SPA_ZERO_POINT_ZERO)
       {
           argPHI = SPA_ZERO_POINT_ZERO;
       }
   }

     /*****************************************************************/
     /* Compute argGAMMA (T2) as right side of half-angle formula for */
     /* rotation angle GAMMA                                          */
     /* = sin(sigma-theta)*sin(sigma-phi)/(sin(theta)*sin(phi))       */
     /* (October 23 Memo, top of page 5) & limit between 0.0 & 1.0    */
     /*****************************************************************/

   argGAMMA =  sin( sigma - theta ) * sin( sigma - phi ) /
             ( sin(         theta ) * sin(         phi ) );

   if (argGAMMA > SPA_ONE_POINT_ZERO)
   {
       argGAMMA = SPA_ONE_POINT_ZERO;
   }
   else
   {
       if (argGAMMA < SPA_ZERO_POINT_ZERO)
       {
           argGAMMA = SPA_ZERO_POINT_ZERO;
       }
   }

                 /***********************************/
                 /***********************************/
                 /* Evaluate inverse trig functions */
                 /***********************************/
                 /***********************************/

            /**********************************************************/
            /* Compute PHI as twice the arcsine of the square root of */
            /* argPHI(T1).  Compute GAMMA as twice the arcsine of the */
            /* square root of argGAMMA(T2).  (Since the argument was  */
            /* bounded 0 to 1, this function will work)               */
            /**********************************************************/

   PHI = SPA_TWO_POINT_ZERO * asin(sqrt(argPHI));
   
   GAMMA = SPA_TWO_POINT_ZERO * asin(sqrt(argGAMMA));


   /****************************************************************/
   /****************************************************************/
   /* Compute corrections for Earth rotation and orbit precession: */
   /* ------------------------------------------------------------ */
   /* Compute tr, the approximate orbit time to progress the       */
   /* rotation angle GAMMA.  Compute delta, the Earth rotation in  */
   /* this time.  Compute epsilon, a correction for orbit          */
   /* precession in this time.                                     */
   /****************************************************************/
   /****************************************************************/

   tr = orbit_period * (GAMMA * RAD2DEG / SPA_THREE_SIXTY_POINT_ZERO );

   delta = earth_rate * tr;
 
   epsilon = orbit_precession * tr / orbit_period;
  

               /*******************************************************/
               /* The relative longitude result is computed from PHI, */
               /* corrected for Earth rotation and orbit precession.  */
               /*******************************************************/
 
   *lon = SPA_ONE_EIGHTY_POINT_ZERO - PHI * RAD2DEG - delta + epsilon;

}
