/* CVS Header Section
 * 
 * $Log: SPAgetModParams.c,v $
 * Revision 1.1.1.1  2011/12/30 02:47:49  hensley
 * First import
 *
 * 
 * Removed includes of stdio.h, math.h, and string.h
 * Replaced calls to UTmessage with reporting the error to standard out
 * 
 * Checked into sturmmitte CVS as "spacomlin" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * Revision 1.1.1.1  2011/04/08 01:37:24  hensley
 * Import for V7 OAT
 * 
 * Checked into sturmmitte CVS as "spav7" ^^^ 
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * Revision 1.1.1.1  2009/08/13 20:18:16  hensley
 * First added
 * 
 * Revision 1.2  2009/07/02 20:43:09  hensley
 * TSDIS to PPS, 72 chars wide, UTreportWarning to UTmessage
 * 
 * Revision 1.1.1.1  2008/06/18 23:06:08  hensley
 * Baseline PPS-- Code
 * 
 * Linux CVS comments when imported as "spa" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * SGI CVS Comments vvv
 * 
 * Revision 1.5  2005/12/13 18:45:02  hensley
 * Fixed the CVS Header Section
 * 
 * Revision 1.4  1997/08/01 17:40:07  piper
 * Removed unused variable (k).
 * 
 * Revision 1.3  1997/03/26 08:15:47  bilanow
 * Flattening coefficient tolerance assignment fixed.
 * 
 * Revision 1.2  1996/11/22  22:04:39  bilanow
 * added #include <string.h>
 * 
 * Revision 1.1  1996/11/21  22:24:38  bilanow
 * New routine for Spatial Search to read parameters
 * file, added to CVS first time.
 * 
 */

/***********************************************************************
@ Function Statement:
@ -------------------
@   int SPAgetModParams(char  *fileID,
@                       float *orbit_inclination,
@                       float *kilometers_per_deg,
@                       float *flatcoeff,
@                       float *orbit_period,
@                       float *orbit_precession,
@                       float *earth_rate)
@
@  Description:
@  ------------
@              This function reads a fixed-format file to get the swath
@              model parameters.  Default values for these parameters
@              are stored as constants (in SPA.h), however modifications
@              to these parameters are read from the file.  PPS system
@              warning messages are provided if problems are encountered
@              reading the file, or if the input values are outside the
@              reasonable tolerances (which are also stored as constants
@              in SPA.h).
@
@
@ Input:    fileID               -  swath model parameters file
@ ------
@
@ Output:                           (and notes on usage)
@ -------
@           orbit_inclination    -  The TRMM orbit inclination (deg)
@           kilometers_per_deg   -  km/degree for swath width model
@                                   (The above two are used to compute
@                                   inputs to SPAgetBVal which give
@                                   sides of a spherical triangle for
@                                   the swath boundary computation)
@           flatcoeff            -  Earth oblateness flattening
@                                   coefficient (used in geodetic to
@                                   geocentric latitude conversion in
@                                   SPAspatialBound)
@           orbit_period         -  The mean TRMM orbit period (seconds)
@           orbit_precession     -  The mean TRMM orbit precession rate
@                                   (degrees/orbit)
@           earth_rate           -  The earth rotation rate (deg/sec)
@                                   (The above three are used in
@                                   SPAgetBVal to help define the swath
@                                   edges on the Earth)
@
@ Return:   
@ -------
@ status 
@ 
@ TS_SUCCESS                        normal return, valid values read
@                                   from file
@
@ For all of the following status returns, all default values are
@ returned with output:
@
@ W_SPA_MOD_PARAMS_FILE_NOT_FOUND   Error in opening file occurred
@ W_SPA_MOD_PARAMS_FILE_INCOMPLETE  Error in reading file occurred
@ W_SPA_MOD_PARAMS_MISSING_COLON    Colon not found at expected column
@ W_SPA_MOD_PARAM_ORBIT_INC_ERROR   Out-of-tolerance orbit_inclination
@                                   read
@ W_SPA_MOD_PARAM_KILO_2_DEG_ERROR  Out-of-tolerance kilometers_per_deg
@                                   read
@ W_SPA_MOD_PARAM_FLATCOEFF_ERROR   Out-of-tolerance flatcoeff read
@ W_SPA_MOD_PARAM_ORB_PERIOD_ERROR  Out-of-tolerance orbit_period read
@ W_SPA_MOD_PARAM_ORB_PREC_ERROR    Out-of-tolerance orbit_precession
@                                   read
@ W_SPA_MOD_PARAM_EARTH_RATE_ERROR  Out-of-tolerance earth_rate read
@ 
@
@ Called By:  SPAspatialBound
@ ----------
@
@ Routines called:  None
@ ----------------
@
@
@   Date    Programmer Name   Change Description 
@ --------  ---------------   ------------------------------------------
@ 96-11-19  Steve Bilanow     New:  Obtains parameters from file 
@
@ 97-03-25  Steve Bilanow     Tolerance on flattening coefficient fixed
@
@ 08-06-18  Michael Hensley   Added fileID to the input parameters
@ 
@ 09-07-02  Michael Hensley   Changed calls to UTreportWarning to
@                             UTmessage
@ 11-12-28  Michael Hensley   Replaced calls to UTmessage with reporting
@                             the error to standard out, removed
@                             stdio.h, math.h, and string.h includes
@
@***********************************************************************
@ Algorithm (PDL)
@ ---------------
@
@   Assign Default Values (from constants)
@
@   Open file
@     if error,
@        Write error message to standard out
@        return
@     endif
@
@   For each line of text read (12 lines)
@
@      Read in line of text
@        if error,
@           Write error message to standard out
@           return
@        endif
@
@        if parameter line and colon not found in proper column,
@           Write error message to standard out
@           return
@        endif
@
@        make sure null character marks end of string for each line
@
@   End For, finish reading needed text lines
@
@   For parameter 1 
@       Convert string to float stored as local variable
@        If input value separation from default value exceeds tolerance
@           Write error message to standard out
@           return
@        endif
@
@   For parameter 2
@       Convert string to float stored as local variable
@        If input value separation from default value exceeds tolerance
@           Write error message to standard out
@           return
@        endif
@
@   For parameter 3
@       Convert string to float stored as local variable
@        If input value separation from default value exceeds tolerance
@           Write error message to standard out
@           return
@        endif
@
@   For parameter 4
@       Convert string to float stored as local variable
@        If input value separation from default value exceeds tolerance
@           Write error message to standard out
@           return
@        endif
@
@   For parameter 5
@       Convert string to float stored as local variable
@        If input value separation from default value exceeds tolerance
@           Write error message to standard out
@           return
@        endif
@
@   For parameter 6 
@       Convert string to float stored as local variable
@        If input value separation from default value exceeds tolerance
@           Write error message to standard out
@           return
@        endif
@
@   Assign output values from local values converted from input string
@
@   return status as success
@
@**********************************************************************/

/************/
/* Includes */
/************/

#include "SPAproto.h"

/**************************************************
* Constants                                       *
*                                                 *
* Note that default constants for the swath model *
* and tolerances are defined in the SPA.h file.   *
*                                                 *
* Local constants for this routine define the     *
* model parameters file name and the lines to be  *
* read and positions of the parameter values.     *
**************************************************/

#define NUMPARMS 12	/* number of lines to read in */
#define STARTLN   6	/* number of header lines     */
#define VALPOS   55	/* column where value starts  */

/************************
*  Function Definition  *
************************/

int SPAgetModParams(char  *fileID,
                    float *orbit_inclination,
                    float *kilometers_per_deg,
                    float *flatcoeff,
                    float *orbit_period,
                    float *orbit_precession,
                    float *earth_rate)

{

   /************************
   * Variable Declarations *
   ************************/

   float value_read;	/* value converted to float from string       */
   float default_val;	/* default value for parameter check          */
   float tolerance_val;	/* tolerance used for parameter check         */

   float local_orbit_inclination;
			/* local variable to store orbit_inclination  */

   float local_kilometers_per_deg;
			/* local variable to store kilometers_per_deg */

   float local_flatcoeff;
			/* local variable to store flatcoeff          */

   float local_orbit_period;
			/* local variable to store orbit_period       */

   float local_orbit_precession;
			/* local variable to store orbit_precession   */

   float local_earth_rate;
			/* local variable to store earth_rate         */

   FILE *mp_fp;		/* Model parameter file pointer               */

   char *path = "./";	/* Path to input file                         */
   char *ptr;		/* Pointer to parameter strings               */

   char fullpath[250];	/* Path/filename of input file                */

   char str[30][100];	/* Container for the parameter file contents  */

   int i;		/* Counter variables (line #, char # )        */
   int len;		/* Parameter string length                    */

   int status = TS_SUCCESS;	/* Return status                      */

   char *errDefVal = "DEFAULT VALUES WILL BE USED";
   char *errMsg001 = "fopen failed.  Could not open the file";
   char *errMsg002 = "has invalid format or got corrupted";
   char *errMsg003 = "Out-of-tolerance orbit_inclination value =";
   char *errMsg004 = "Out-of-tolerance kilometers_per_deg value =";
   char *errMsg005 = "Out-of-tolerance flatcoeff value =";
   char *errMsg006 = "Out-of-tolerance orbit_period value =";
   char *errMsg007 = "Out-of-tolerance orbit_precession value =";
   char *errMsg008 = "Out-of-tolerance earth_rate value =";

   /****************************************************/
   /* Assign default values: In case errors are found, */
   /* these are used.                                  */
   /****************************************************/

   *orbit_inclination  = SPA_SWATH_MODEL_ORBIT_INCLINATION;
   *kilometers_per_deg = SPA_SWATH_MODEL_KILOMETERS_PER_DEGREE;
   *flatcoeff          = SPA_SWATH_MODEL_FLATCOEFF;
   *orbit_period       = SPA_SWATH_MODEL_ORBIT_PERIOD;
   *orbit_precession   = SPA_SWATH_MODEL_ORBIT_PRECESSION;
   *earth_rate         = SPA_SWATH_MODEL_EARTH_RATE;

   /***************************************************
   * Open the model parameter input file, using the   *
   * hard coded file name defined, and if error,      *
   * write an appropriate message to standard out.    *
   ***************************************************/

   strcpy(fullpath, path);
   strcat(fullpath, fileID);

   if ((mp_fp = fopen(fullpath, "r")) == NULL)
   {

      printf("\n\n%s %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg001, fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAMS_FILE_NOT_FOUND;
      return status;

   }

   /***********************************************************
   * Read each line needed into a string array and if error,  *
   * write an appropriate message to standard out.  After the *
   * data starting line is passed, if a colon is missing at   *
   * the appropriate column, write an appropriate messate to  *
   * standard out.  Return with error flag if any error is    *
   * encountered.                                             *
   ***********************************************************/

   for( i=0 ; i<NUMPARMS ; i++ )
   {

      if ((fgets (str[i], 85, mp_fp)) == NULL)
      {

         printf("\n\n%s \"%s\" %s.  %s.  %s %s,  %s %d\n\n",
                "WARNING: fgets failed.  The file",
                fileID, errMsg002, errDefVal,
                "File:", __FILE__, "Line #:", __LINE__);

         status = W_SPA_MOD_PARAMS_FILE_INCOMPLETE;
         return status;

      }

      if (i >= STARTLN && str[i][VALPOS-1] != ':')
      {

         /*************************************/
         /* Model parameter file format error */
         /*************************************/

         printf("\n\n%s \"%s\" %s.  %s.  %s %s,  %s %d\n\n",
                "WARNING: Colon missed.  The file",
                fileID, errMsg002, errDefVal,
                "File:", __FILE__, "Line #:", __LINE__);

         status = W_SPA_MOD_PARAMS_MISSING_COLON;
         return status;

      }

      /**********************************/
      /* Put null char on end of string */
      /**********************************/

      len = (int)strlen(str[i]) ;
      str[i][len-1] = '\0';

   }

   /*******************************************
   * Close the Model parameters input file.   *
   * All needed text is now in string arrays. *
   *******************************************/

   fclose(mp_fp);

   /*****************************************************
   * Extract the values of each of the model parameters *
   *****************************************************/

   i = STARTLN;

   /********************
   * orbit_inclination *
   ********************/

   ptr = &str[i][VALPOS];
   value_read    = atof(ptr);
   default_val   = SPA_SWATH_MODEL_ORBIT_INCLINATION;
   tolerance_val = SPA_SWATH_MODEL_ORBIT_INCLINATION_TOL;

   if ( fabs( value_read - default_val) > tolerance_val )
   {

      printf("\n\n%s %s %f.  %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg003, value_read, "Check the file",
             fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAM_ORBIT_INC_ERROR;
      return status;

   }
 
   local_orbit_inclination = value_read;
   i++;

   /*********************
   * Kilometers_per_deg *
   *********************/

   ptr = &str[i][VALPOS];
   value_read    = atof(ptr);
   default_val   = SPA_SWATH_MODEL_KILOMETERS_PER_DEGREE;
   tolerance_val = SPA_SWATH_MODEL_KILOMETERS_PER_DEGREE_TOL;

   if ( fabs( value_read - default_val) > tolerance_val )
   {

      printf("\n\n%s %s %f.  %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg004, value_read, "Check the file",
             fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAM_KILO_2_DEG_ERROR;
      return status;

   }

   local_kilometers_per_deg = value_read;
   i++;
 
   /************
   * flatcoeff *
   ************/

   ptr = &str[i][VALPOS];
   value_read    = atof(ptr);
   default_val   = SPA_SWATH_MODEL_FLATCOEFF;
   tolerance_val = SPA_SWATH_MODEL_FLATCOEFF_TOL;

   if ( fabs( value_read - default_val) > tolerance_val )
   {

      printf("\n\n%s %s %f.  %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg005, value_read, "Check the file",
             fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAM_FLATCOEFF_ERROR;
      return status;

   }

   local_flatcoeff = value_read;
   i++;
 
   /*************** 
   * orbit_period *
   ***************/

   ptr = &str[i][VALPOS];
   value_read    = atof(ptr);
   default_val   = SPA_SWATH_MODEL_ORBIT_PERIOD;
   tolerance_val = SPA_SWATH_MODEL_ORBIT_PERIOD_TOL;

   if ( fabs( value_read - default_val) > tolerance_val )
   {

      printf("\n\n%s %s %f.  %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg006, value_read, "Check the file",
             fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAM_ORB_PERIOD_ERROR;
      return status;

   }

   local_orbit_period = value_read;
   i++;
 
   /******************* 
   * orbit_precession *
   *******************/

   ptr = &str[i][VALPOS];
   value_read    = atof(ptr);
   default_val   = SPA_SWATH_MODEL_ORBIT_PRECESSION;
   tolerance_val = SPA_SWATH_MODEL_ORBIT_PRECESSION_TOL;

   if ( fabs( value_read - default_val) > tolerance_val )
   {

      printf("\n\n%s %s %f.  %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg007, value_read, "Check the file",
             fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAM_ORB_PREC_ERROR;
      return status;

   }

   local_orbit_precession =  value_read;
   i++;
 
   /************* 
   * earth_rate *
   *************/

   ptr = &str[i][VALPOS];
   value_read    = atof(ptr);
   default_val   = SPA_SWATH_MODEL_EARTH_RATE;
   tolerance_val = SPA_SWATH_MODEL_EARTH_RATE_TOL;

   if ( fabs( value_read - default_val) > tolerance_val )
   {

      printf("\n\n%s %s %f.  %s \"%s\".  %s.  %s %s,  %s %d\n\n",
             "WARNING:", errMsg008, value_read, "Check the file",
             fileID, errDefVal,
             "File:", __FILE__, "Line #:", __LINE__);

      status = W_SPA_MOD_PARAM_EARTH_RATE_ERROR;
      return status;

   }

   local_earth_rate =  value_read;

   /********************************************
   * Assign values that have all passed checks *
   ********************************************/

   *orbit_inclination  = local_orbit_inclination;
   *kilometers_per_deg = local_kilometers_per_deg;
   *flatcoeff          = local_flatcoeff;
   *orbit_period       = local_orbit_period;
   *orbit_precession   = local_orbit_precession;
   *earth_rate         = local_earth_rate;

   return status;

}
