/* CVS Comment Section
 * 
 * $Log: SPAspatialBound.c,v $
 * Revision 1.1.1.1  2011/12/30 02:47:49  hensley
 * First import
 *
 * 
 * Checked into sturmmitte CVS as "spacomlin" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * Revision 1.3  2011/07/01 22:21:19  hensley
 * Changed TSDIS to PPS
 * 
 * Revision 1.2  2008/08/20 21:28:19  hensley
 * Removed stdio.h and math.h includes
 * 
 * Revision 1.1.1.1  2008/08/19 21:17:02  hensley
 * First Import
 * 
 * Checked into sturmmitte CVS as "spaweb" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * Revision 1.1.1.1  2008/06/18 23:06:08  hensley
 * Baseline PPS-- Code
 * 
 * Linux CVS Comments When Imported as "spa" ^^^
 * ---------------------------------------------------------------------
 * ---------------------------------------------------------------------
 * SGI CVS Comments vvv
 * 
 * Revision 1.3  2005/12/13 18:45:12  hensley
 * Fixed the CVS Header Section
 * 
 * Revision 1.2  1996/11/22 02:06:35  bilanow
 * Post-walkthrough modifications to get parameters from file, etc.
 * 
 * Revision 1.1  1996/11/14  18:53:25  dbazell
 * moved file from spat/ to spa/
 * 
 * Revision 1.1  1996/09/20  19:33:10  bilanow
 * Added to CVS first time.
 * 
 **/

/***********************************************************************
@ Function Statement:
@ -------------------
@  int SPAspatialBound (char  *fileID,  float swathw,
@                       float latmin,   float latmax,
@                       float lonmin,   float lonmax, 
@                       float *ascmin1, float *ascmax1, 
@                       float *desmin1, float *desmax1,
@                       float *ascmin2, float *desmin2)
@ 
@ Description:
@ ------------
@ 
@      This function returns the search boundaries for the database
@      parameters to accomplish a spatial search for swath intersections
@      with any selected latitude-longitude box, and any selected swath
@      width.  When the database is searched according to the boundaries
@      provided for the metadata parameter Longitude_of_Max_Latitude, it
@      will return all orbits where the input swath width would
@      intersect the input latitude-longitude box.
@ 
@      The algorithm implemented here is described in a memo dated
@      10-23-95 from Steve Bilanow regarding: An Efficient Search
@      Algorithm for Coincidence of Swath Granules w/ Latitude/Longitude
@      Spatial Coverage.  That memo was an update of the earlier memo
@      describing the algorithm which was dated June 30, 1995.
@ 
@      Prototype code was developed by Udaya Adusumalli in October, 1995
@      based on the detailed algorithm analysis and the top level API
@      defined by Steve Bilanow.  PDL was developed by November 1995
@      based on the prototype & algorithm refinements by Steve Bilanow.
@ 
@ Input:  
@ ------
@      fileID - fully qualified path/filename for the parameters file
@      swathw - The width of the swath in kilometers
@               (For PPS, approximate values for each sensor are
@                           PR  =  215 km
@                           VIRS = 720 km
@                           TMI  = 760 km )
@ 
@      latmin - Minimum latitude of box in degrees (+/- 90, + is North)
@      latmax - Maximum latitude of box in degrees (+/- 90, + is North)
@ 
@      lonmin - Minimum longitude of box in degrees (+/- 360, + is East)
@      lonmax - Maximum longitude of box in degrees (+/- 360, + is East)
@ 
@      (Note that lonmin can be less that -180 or lonmax can be greater
@       than +180 so that boxes can be readily specified that span the
@       international date line)
@ 
@ Output:
@ -------
@      ascmin1 - ascending portion minimum Longitude-of-Max-Latitude for
@                coverage
@      ascmax1 - ascending portion maximum Longitude-of-Max-Latitude for
@                coverage
@ 
@           i.e., coverage occurs if:
@ 
@                ascmin1 < Longitude-of-Max-Latitude < ascmax1
@ 
@      desmin1 - descending portion minimum Longitude-of-Max-Latitude for
@                coverage
@      desmax1 - descending portion maximum Longitude-of-Max-Latitude for
@                coverage
@ 
@           i.e., coverage occurs if:
@ 
@                desmin1 < Longitude-of-Max-Latitude < desmax1
@ 
@      ascmin2 - second boundary value for minimum
@                Longitude-of-Max-Latitude to be used in case the range
@                for the ascending portion must be split in two parts to
@                span the international date line around 180 longitude.
@                If not needed, this is set to +999.0.  If needed,
@                ascmin1 will be less than or equal to -180, ascmax1
@                will be greater than -180, ascmin2 will be greater than
@                ascmax1 and less than 180, and the search for the
@                database on Longitude-of-Max-Latitude should check both
@                the following ranges:
@ 
@                      -180 < Longitude-of-Max-Latitude < ascmax1, and
@                   ascmin2 < Longitude-of-Max-Latitude < +180
@ 
@      desmin2 - second boundary value for minimum
@                Longitude-of-Max-Latitude to be used in case the range
@                for the descending portion must be split in two parts
@                to span the international date line around 180
@                longitude.  If not needed, this is set to +999.0.  If
@                needed, desmin1 will be less than or equal to -180,
@                desmax1 will be greater than -180, desmin2 will be
@                greater than desmax1 and less than 180, and the search
@                for the database on Longitude-of-Max-Latitude should
@                check both the following ranges:
@ 
@                      -180 < Longitude-of-Max-Latitude < desmax1, and
@                   desmin2 < Longitude-of-Max-Latitude < +180
@ 
@      (Note that the above parameters are provided assuming that the
@       Longitude-of-Max-Latitude is stored as a value between + & -
@       180)
@ 
@ Return:
@ -------
@       Error Codes as follows:   =0, normal return (success)
@                                 =1, swathw < 0
@                                 =2, latmax < latmin
@                                 =3, lonmax < lonmin
@                                 =4, Input Lat-Lon box outside swath
@                                     latitude range.
@ 
@ Routines called:  SPAgetBVal      - Evaluates swath boundaries versus
@                                     geocentric latitude
@                   SPAgetModParams - reads in swath model parameters
@ 
@   Date     Programmer Name   Change Description 
@   ----     --------------    -------------------------
@ 96-09-06   Steve Bilanow     Prototype Enhacement to Initial Build
@                              Version 
@ 
@ 08-06-18   Michael Hensley   Added fileID to the input parameters and
@                              to the call to SPAgetModParams
@ 
@***********************************************************************
@ Algorithm (PDL)
@ ---------------
@ 
@ Set default outputs in case errors are encountered as checked below,
@ or the second output boundary value isn't needed.  If used anyway in
@ a database search, these default outputs would provide the correct
@ answer of no coverage.
@ 
@     ascmin1 = -999.0
@     ascmax1 = -999.0
@     desmin1 = -999.0
@     desmax1 = -999.0
@     ascmin2 =  999.0
@     desmin2 =  999.0
@ 
@ Check for error codes as follows:
@ --------------------------------
@ 
@ IF the input swath width is less than zero
@ THEN 
@     Return with Error Code 1
@ ENDIF
@ 
@ IF the input maximum latitude is greater than the input minimum
@    latitude
@ THEN
@     Return with Error Code 2
@ ENDIF
@ 
@ IF the input maximum longitude is less than the input minimum
@    longitude
@ THEN
@     Return with Error Code 3
@ ENDIF
@ 
@ 
@ Get Swath Model Parameters from File
@ ------------------------------------
@  
@ CALL SPAgetModParams to get swath-shape-related parameters
@ INPUTS:   fileID,             swath model parameters file
@ OUTPUTS:  orbit_inclination,  orbit inclination (deg)
@           kilometers_per_deg, (km/deg)
@           flatcoeff,          Earth oblateness coefficient
@           orbit_period,       orbit period (sec)
@           orbit_precession,   orbit precession (deg/orbit)
@           earth_rate,         Earth rotation rate (deg/sec)
@ RETURN:  statusParams, flag indicating if default values used
@ 
@ Set return status for SPAspatialBound if default values used.
@ 
@ 
@ Use SPAgetBVal to evaluate boundary functions as needed:
@ ----------------------------------------------------
@ 
@ Convert input latitudes in degrees to equivalent geodetic latitude
@ (radians)
@ 
@ Compute input parameters to getBVal in radians
@ (theta as constant orbit inclination in radians,
@  phi as the pi/2 minus the swath half width in radians, and
@  gamma as the colatitude of the bottom of the box, in radians)
@ 
@ CALL SPAgetBVal to evaluate the upper boundary function for the
@      minumum latitude
@ INPUTS:  theta, Orbit inclination (rad)
@          phi,   Arc from Orbit Normal to the swath upper bound (rad)
@          gamma, Arc from the Earth Pole to the minimum latitude (rad)
@ OUTPUTS:  Upper boundary function evaluated for the minumum latitude
@           (deg) (upper_min)
@ 
@ Compute input parameters to getBVal in radians
@ (phi as the pi/2 plus the swath half width in radians, and
@  gamma as the colatitude of the top of the box, in radians)
@ 
@ 
@ CALL SPAgetBVal to evaluate the lower boundary function for the
@      maximum latitude
@ INPUTS:  theta, Orbit inclination (rad)
@          phi,   Arc from Orbit Normal to the swath lower bound (rad)
@          gamma, Arc from the Earth Pole to the maximum latitude (rad)
@ OUTPUTS: Lower boundary function evaluated for the maximum latitude
@          (deg) (lower_max)
@ 
@ Check for input latitude/longitude box outside the swath latitude
@ range:
@ -----------------------------------------------------------------
@ 
@ IF Upper and Lower boundary values obtained are the same
@ THEN
@     Return with Error Code 4
@ ENDIF
@ 
@ Obtain basic boundary conditions for database search:
@ ----------------------------------------------------
@ 
@ Compute descending portion minimum Longitude-of-Max-Latitude for
@         coverage (desmin1 = lonmin - upper_min)
@ Compute descending portion maximum Longitude-of-Max-Latitude for
@         coverage (desmax1 = lonmax - lower_max)
@ Compute ascending portion minimum Longitude-of-Max-Latitude for
@         coverage (ascmin1 = lonmin + lower_max)
@ Compute ascending portion maximum Longitude-of-Max-Latitude for
@         coverage (ascmax1 = lonmax + upper_min)
@ 
@ Make sure search upper boundary conditions are in -180 to +180 range:
@ --------------------------------------------------------------------
@ 
@ IF descending portion Maximum Longitude-of-Max-Latitude is less than
@    -180,
@ THEN
@     Add 360 to descending portion Minimum Longitude-of-Max-Latitude
@        (desmin1)
@     Add 360 to descending portion Maximum Longitude-of-Max-Latitude
@        (desmax1)
@ ENDIF
@ 
@ IF descending portion Maximum Longitude-of-Max-Latitude is greater
@    than +180
@ THEN
@     Subtract 360 from descending portion Minimum
@        Longitude-of-Max-Latitude (desmin1)
@     Subtract 360 from descending portion Maximum
@        Longitude-of-Max-Latitude (desmax1)
@ ENDIF
@ 
@ IF ascending portion Maximum Longitude-of-Max-Latitude is less than
@    -180,
@ THEN
@     Add 360 to ascending portion Minimum Longitude-of-Max-Latitude
@        (ascmin1)
@     Add 360 to ascending portion Maximum Longitude-of-Max-Latitude
@        (ascmax1)
@ ENDIF
@ 
@ IF ascending portion Maximum Longitude-of-Max-Latitude is greater than
@    +180
@ THEN
@     Subtract 360 from ascending portion Minimum
@        Longitude-of-Max-Latitude (ascmin1)
@     Subtract 360 from ascending portion Maximum
@        Longitude-of-Max-Latitude (ascmax1)
@ ENDIF
@ 
@ Handle division of boundary conditions into two pieces if needed to
@ span date line:
@ -------------------------------------------------------------------
@ 
@ IF descending portion Minimum Longitude-of-Max-Latitude is less than
@    -180
@ THEN
@     Set second descending portion Minimum Longitude-of-Max-Latitude
@        (desmin2) equal to descending portion Minimum
@        Longitude-of-Max-Latitude (desmin1) plus 360
@ ENDIF
@ 
@ IF ascending portion Minimum Longitude-of-Max-Latitude is less than
@    -180
@ THEN
@     Set second ascending portion Minimum Longitude-of-Max-Latitude
@        (ascmin2) equal to ascending portion Minimum
@        Longitude-of-Max-Latitude (ascmin1) plus 360
@ ENDIF
@ 
@ Done
@ ----
@ 
@ Return with status flag to Calling routine
@ 
***********************************************************************/

#include "SPAproto.h"

        /****************************************************
        *   Notes that key constants for the swath model    *
        *   are defined in the SPA.h file,                  *
        *   which is included in the SPAproto.h file.       *
        ****************************************************/

int SPAspatialBound (char  *fileID,  float swathw,
                     float latmin,   float latmax,
                     float lonmin,   float lonmax,
                     float *ascmin1, float *ascmax1,
                     float *desmin1, float *desmax1,
                     float *ascmin2, float *desmin2)

{

                      /****************************
                      *  Variable Declarations    *
                      ****************************/

                               /* Six swath model parameters:         */
                               /* (obtained by SPAgetModParams)       */
   float orbit_inclination;    /*  orbit inclination in degrees       */
   float kilometers_per_deg;   /*  kilometers per degree for swath    */
   float flatcoeff;            /*  Earth oblateness coefficient       */
   float orbit_period;         /*  orbit period in seconds            */
   float orbit_precession;     /*  orbit precession rate in deg/orbit */
   float earth_rate;           /*  Earth rotation rate in deg/sec     */


   float omf2;    /* (One minus the Earth flattening coefficient)     */
                  /* squared.  This term is used in geodetic to       */
                  /* geocentric latitude conversions.                 */

   float geoclatmax;          /* geocentric latitude of top of box    */ 
   float geoclatmin;          /* geocentric latitude of bottom of box */

   float swathwDegrees;       /* swath width in degrees of arc        */

                          /****************************************/
                          /* Note theta, phi, and gamma are sides */
                          /* of a spherical triangle in inertial  */
                          /* space used for the swath model       */
                          /****************************************/

   float theta;           /* orbit inclination (radians)              */
   float phi_up;          /* arc from orbit normal to swath "top"     */
   float phi_lo;          /* arc from orbit normal to swath "bottom"  */
   float gama_max;        /* arc from the pole to the box top         */
   float gama_min;        /* arc from the pole to the box bottom      */

   float upper_min;       /* angle relative to lon_of_max_lat where   */
                          /* upper boundary of the swath hits the     */
                          /* lower edge of the input box              */

   float lower_max;       /* angle relative to lon_of_max_lat where   */
                          /* lower boundary of the swath hits the     */
                          /* upper edge of the input box              */

   int statusParams;

   int status = TS_SUCCESS;  /* Return status, nominally = TS_SUCCESS */

/**********************************************************************
* Set default outputs in case errors are encountered as checked below,* 
* or the second output boundary value isn't needed.                   *
* If used anyway in a database search, these default outputs would    *
* provide the correct answer of no coverage.                          *
**********************************************************************/

   *ascmin1 = SPA_DEFAULT_ASCMIN1;
   *ascmax1 = SPA_DEFAULT_ASCMAX1;
   *desmin1 = SPA_DEFAULT_DESMIN1;
   *desmax1 = SPA_DEFAULT_DESMAX1;
   *ascmin2 = SPA_DEFAULT_ASCMIN2;
   *desmin2 = SPA_DEFAULT_DESMIN2;

                                        /*******************************
                                        *  Check for errors in inputs  *
                                        *  and return with error codes *
                                        *******************************/

   if (swathw < SPA_ZERO_POINT_ZERO)      /* negative swath width     */
      return SPA_RETURN1_NEG_SWATH_WIDTH;

   if (latmax < latmin)                   /* box min > max latitude   */
      return SPA_RETURN2_MIN_GT_MAX_LAT;

   if (lonmax < lonmin)                   /* box min > max longitude  */
      return SPA_RETURN3_MIN_GT_MAX_LON;

                                       /********************************
                                       * Use SPAgetModParams to obtain *
                                       * swath-shape-related model     *
                                       * parameters from input file.   *
                                       * If any return but SUCCESS,    *
                                       * then SPAspatialBound status   *
                                       * returned will indicate that   *
                                       * default parameters are used.  *
                                       ********************************/

   statusParams = SPAgetModParams(fileID,
                                  &orbit_inclination,
                                  &kilometers_per_deg,
                                  &flatcoeff,
                                  &orbit_period,
                                  &orbit_precession,
                                  &earth_rate);

   if ( statusParams != TS_SUCCESS )
      status = SPA_RETURN5_DEFAULT_MODEL_PARAMS;

                                        /*******************************
                                        *  Use SPAgetBVal to obtain    *
                                        *  boundary conditions needed  *
                                        *  on Lon-of-Max-Latitude      *
                                        *******************************/

             /**********************************************************
             * Convert input latitudes, assumed as geodetic latitudes, *
             * to geocentic latitudes for use in getting the angles    *
             * gama_max and gama_min for input to the swath boundary   *
             * value function evaluation.                              *
             **********************************************************/

   omf2 = ( SPA_ONE_POINT_ZERO - flatcoeff ) *
          ( SPA_ONE_POINT_ZERO - flatcoeff );

   geoclatmax = ( atan ( omf2 * tan ( latmax * DEG2RAD ) ) ) / DEG2RAD;
   geoclatmin = ( atan ( omf2 * tan ( latmin * DEG2RAD ) ) ) / DEG2RAD;

/***********************************************************************
* Compute input parameters to SPAgetBVal:                              *
* i.e. theta as constant orbit inclination,                            *
*        (note that a special SPA specific constant is used to convert *
*        the input swathwidth in kilometers to degrees of arc)         *
*      phi as 90 degrees plus or minus the swath half width,           *
*          i.e phi_up is the arc to the "top" of the swath boundary,   *
*              phi_lo is the arc to the "bottom" of the swath boundary,*
*      gamma as the colatitude of the bottom of the box,               *
*          i.e., gama-max is the arc from the pole to the box top,     * 
*                gama-min is the arc from the pole to the box bottom,  *
*          and all of the above are sent to SPAgetBVal in radians.     *
***********************************************************************/
/***********************************************************************
* Diagram best shown using non-proportional font:                      *
*                                                       Earth          *
* Note:  The geometry for                          x    Pole           *
*        the SPAgetBVal                           / \                  *
*        swath boundary                          /   \  theta          *
*        computation uses              gamma    /     x                *
*        a spherical triangle                  /     /   Orbit Normal  *
*        with sides theta                     /   /                    *
*        gamma and phi.                      //     phi                *
*                                           x                          *
*                                    swath edge point                  *
***********************************************************************/

   theta = orbit_inclination * DEG2RAD;

   swathwDegrees = swathw / kilometers_per_deg;

   phi_up = ( SPA_NINETY_POINT_ZERO - swathwDegrees /
              SPA_TWO_POINT_ZERO ) * DEG2RAD;
   phi_lo = ( SPA_NINETY_POINT_ZERO + swathwDegrees /
              SPA_TWO_POINT_ZERO ) * DEG2RAD;

   gama_max = (SPA_NINETY_POINT_ZERO - geoclatmax ) * DEG2RAD;
   gama_min = (SPA_NINETY_POINT_ZERO - geoclatmin ) * DEG2RAD;

  /*********************************************************************
  *  Upper boundary of the swath intersects the lower edge of the box  *
  *  at the value upper_min, which gives the relative longitude        *
  *  (longitude displacement relative to longitude-of-max-latitude)    *
  *  above which the box misses the swath.                             *
  *  U(swathw;latmin) i.e., U(w;Lat1) as given in 10/23/95 memo        *
  *********************************************************************/
   
   SPAgetBVal(theta,  phi_up,  gama_min, 
              orbit_period, orbit_precession, earth_rate, &upper_min);

  /*********************************************************************
  *  Lower boundary of the swath intersects the upper edge of the box  *
  *  at the value lower_max, which gives the relative longitude        *
  *  (longitude displacement relative to longitude-of-max-latitude)    *
  *  below which the box misses the swath.                             *
  *  L(swathw;latmax) i.e., L(w;Lat2) as given in 10/23/95 memo        *
  *********************************************************************/
   
   SPAgetBVal(theta,  phi_lo,  gama_max, 
              orbit_period, orbit_precession, earth_rate, &lower_max);

                                       /********************************
                                        * Check if condition indicates *
                                        * input Box is above or below  *
                                        * the swath range in latitude  *
                                        * and return with error code.  *
                                        *******************************/

   if ( upper_min == lower_max ) 
      return SPA_RETURN4_BOX_N_OR_S_SWATH;

       /******************************************************
       *  Get ranges for the Longitude of Maximum Latitude   *
       *  for box intersections with swath,                  *
       *  for the descending segment of the swath.....       *
       ******************************************************/

   *desmin1 = lonmin - upper_min;
   *desmax1 = lonmax - lower_max;

       /******************************************************
       *  .....and for the ascending segment of the swath.   *
       ******************************************************/

   *ascmin1 = lonmin + lower_max;
   *ascmax1 = lonmax + upper_min;

                               /****************************************
                               *  Make sure upper boundary conditions  *
                               *  are in the normal range for the      *
                               *  Longitude-of-Maximum-Latitude        *
                               *  (as stored in the PPS database),     *
                               *  which is in the -180 to +180 range.  *
                               ****************************************/

   if (*desmax1 < SPA_LON_OF_MAX_LAT_MIN)
   {
       *desmin1 += SPA_THREE_SIXTY_POINT_ZERO;
       *desmax1 += SPA_THREE_SIXTY_POINT_ZERO;
   }
   if (*desmax1 > SPA_LON_OF_MAX_LAT_MAX)
   {
       *desmin1 -= SPA_THREE_SIXTY_POINT_ZERO;
       *desmax1 -= SPA_THREE_SIXTY_POINT_ZERO;
   }
   if (*ascmax1 < SPA_LON_OF_MAX_LAT_MIN)
   {
       *ascmin1 += SPA_THREE_SIXTY_POINT_ZERO;
       *ascmax1 += SPA_THREE_SIXTY_POINT_ZERO;
   }
   if (*ascmax1 > SPA_LON_OF_MAX_LAT_MAX)
   {
       *ascmin1 -= SPA_THREE_SIXTY_POINT_ZERO;
       *ascmax1 -= SPA_THREE_SIXTY_POINT_ZERO;
   }

                        /***********************************************
                        *   Fill alterate minumum values for the       *
                        *   boundary conditions in the case that the   *
                        *   range must be divided in two pieces        *
                        *   to span the date line (+/-180 meridian).   *
                        *   or other dividing point defined for the    *
                        *   Longitude-of-Maximum-Latitude checked.     *
                        ***********************************************/

   if (*desmin1 < SPA_LON_OF_MAX_LAT_MIN)
   {
       *desmin2 =  *desmin1 + SPA_THREE_SIXTY_POINT_ZERO;
   }
   if (*ascmin1 < SPA_LON_OF_MAX_LAT_MIN)
   {
       *ascmin2 = *ascmin1 + SPA_THREE_SIXTY_POINT_ZERO;
   }

           /************************************************************
           *  Return to the calling program.  The status will equal    *
           *  either TS_SUCCESS (as initialized) or will indicate that *
           *  default model parameters were used due to errors in      *
           *  reading the Spatial Search model parameters file.        *
           ************************************************************/

   return status;

}
