/* CVS Comment Section
 * 
 * $Log: findOrbNums.c,v $
 * Revision 1.1  2011/10/19 20:03:09  hensley
 * First import
 *
 * Revision 1.3  2011/10/06 20:30:05  hensley
 * Added leap years through 2040
 * 
 * Revision 1.2  2008/10/16 22:13:14  hensley
 * Paths set for new server configuration
 * 
 * Revision 1.1.1.1  2008/08/20 23:54:54  hensley
 * First Import
 * 
 * ---------------------------------------------------------------------
 * PPS CVS ^^
 * 
 * TSDIS CVS vv
 * ---------------------------------------------------------------------
 * 
 * Revision 1.9  2008/02/07 21:48:12  hensley
 * Added 2008 leap year
 * 
 * Revision 1.8  2002/10/11 21:56:13  hensley
 * Added debugging and separated code into pre- and post-boost
 * calculations
 * 
 * Revision 1.7  2002/02/11 19:59:43  astocker
 * Another CVS test
 * 
 * Revision 1.6  2002/02/11 19:51:32  hensley
 * Test Check In TO CVS
 * 
 * Revision 1.5  2002/01/29 18:47:51  hensley
 * Took / from beginning of lines 2-4
 * 
 * Revision 1.4  2001/09/27 19:32:23  hensley
 * Added Temporary Fix to Account For Longer Orbit Times Post-Boost
 * - May Adjust Later
 * 
 * Revision 1.3  2001/03/08 17:32:11  hensley
 * Added calculation of end time
 * 
 * Revision 1.2  2001/02/26 15:10:31  hensley
 * Adjusted linFitSlope & linFitYint
 * 
 * Revision 1.1.1.1  2000/11/10 23:11:58  hensley
 * Baseline Source Code
 * 
 */

/***********************************************************************
* int findOrbNums (int iyear[2], int imonth[2], int iday[2],
*                  int *numOrbStart, int *numOrbEnd )
* 
* 
* Description:
* -----------
* Gives the orbit number range associated with a start date and an end
* date.
* 
* 
* Input Parameters:
* ----------------
* iyear[]		int [2]		start and end year
* imonth[]		int [2]		start and end month
* iday[]		int [2]		start and end day
* 
* 
* Output Parameters:
* -----------------
* numOrbStart		int *		starting orbit number
* numOrbEnd		int *		ending orbit number
* 
* 
* Return Codes:
* ------------
* 0 - nominal
* 1 - reset starting orbit number to 158
* 2 - reset both starting and ending orbit numbers to 158
* 
* 
* Date		Programmer Name		Change Description
* ----		---------------		------------------
* 15-JUL-98	Steve Bilanow		Initial Version
* 					fit = sec=5478.9967*num-865701.17
* 30-DEC-98	Steve Bilanow		Adjust fit parameters
* 					fit = sec=5479.0319*num-865761.45
* 15-MAR-99	Steve Bilanow		Adjust fit parameters
* 					fit = sec=5479.0527*num-865811.48
* 					(General fit rather than segment
* 					fit is still fine for this appli-
* 					cation.)
* 21-DEC-99	Steve Bilanow		Update to handle leap year and
* 					adjust fit parameters.
* 18-OCT-00	Michael Hensley		Cleaned up and commented more,
* 					replaced redundant code with loop,
* 					added variables i, linFitSlope,
* 					and linFitYint
* 22-FEB-01	Michael Hensley		Adjusted linFitSlope & linFitYint
* 					Added code to set search end time
* 					to end of day
* 08-MAR-01	Michael Hensley		Added calculation of end time to
* 					put end time at end-of-day and to
* 					fix bug (negative tolerance for
* 					end time now set to positive)
* 13-SEP-01     Steve Bilanow           Need fix for altitude change
*                                       Will try temp fix first to
*                                       not take too much time on this now
*                                       (priorities for ACS analysis)
* 11-FEB-02	Michael Hensley		Added this comment to test CVS check
* 					in/out again and again
* 13-OCT-11	Michael Hensley		Reworked include statements
* 					Reworked fractional to integer
* 					orbit number and reset flag setting
* 					code at end of routine
* 					Enhanced header comments
* -------------------------------------------------------------------------
* NOTE:	For future linear fit adjustments, adjust the values of the
* 	variables "linFitSlope2" and "linFitYinti2" set at the beginning of
* 	the code.
* -------------------------------------------------------------------------
* Accuracy note: Linear fit can be off as much as 200 seconds now, so 300
*  		 second tolerance is still O.K. for orbit range.
* -------------------------------------------------------------------------
* 
**************************************************************************/

/*****************/
/* include files */
/*****************/

#include "comOff.h"

/*****************/
/* routine start */
/*****************/

int findOrbNums (int iyear[], int imonth[], int iday[],
                 int *numOrbStart, int *numOrbEnd )
{

/*****************************/
/* local variable definition */
/*****************************/

  int i;			/* loop counter				*/

  int status=0;			/* return status, default zero		*/

  int dayofYear;		/* DOY based on month and DOM		*/

  int daysSince971207;		/* days since 12/7/97			*/

  double secSinceOrb158;	/* seconds since start of orbit 158 for	*/
				/* input date				*/

  double fracOrbnum;		/* fractional orb number for start day	*/

  double sec158to21507 = 116980969.384;
				/* sec from orb 158 - 21507 for		*/
				/* post-boost fit			*/

/*******************************************/
/* define linear fit slope and y-intercept */
/*******************************************/

  double linFitSlope  = 5479.0576; /* orig orb period fit 4 pre-boost */
  double linFitSlope2 = 5543.3398; /* est avg orb period 4 post-boost */

  double linFitYint  =     865785.71; /* orig Y-int fit 4 pre-boost */
  double linFitYint2 = -119220280.00; /* est Y-int fit 4 post-boost */

/*************************************************/
/* define days of year already before each month */
/*************************************************/

  static int dayforMonth[12] = {0,31,59,90,120,151,181,212,243,273,304,334};
  static int dayforMleap[12] = {0,31,60,91,121,152,182,213,244,274,305,335};

/********************************/
/* loop over start and end date */
/********************************/

  for (i=0 ; i<2 ; i++)
  {

/******************************************************************/
/* calculate day of year for years other than 1997 and leap years */
/******************************************************************/

    dayofYear = dayforMonth[imonth[i]] + iday[i];  /* month indices 0-11 */

/****************************************/
/* calculate day of year for leap years */
/****************************************/

    if (iyear[i] == 2000 ||
        iyear[i] == 2004 ||
        iyear[i] == 2008 ||
        iyear[i] == 2012 ||
        iyear[i] == 2016 ||
        iyear[i] == 2020 ||
        iyear[i] == 2024 ||
        iyear[i] == 2028 ||
        iyear[i] == 2032 ||
        iyear[i] == 2036 ||
        iyear[i] == 2040 )
    {
      dayofYear = dayforMleap[imonth[i]] + iday[i];
    }

/**************************************************************/
/* calculate days since 12/7/97 for mission start year (1997) */
/**************************************************************/

    if (iyear[i] == 1997)
    {
      daysSince971207 = dayofYear - 341;
    }
    else  /* calculate days since 12/7/97 for year > 1997 */
    {

/************************************************/
/* calculate days since 12/7/97 for year > 1997 */
/************************************************/

      daysSince971207 = 24 + dayofYear + 365 * (iyear[i] - 1998);

/*************************************************/
/* add leap days to days since 12/7/97 if needed */
/*************************************************/

      if (iyear[i] > 2000) daysSince971207 += 1;
      if (iyear[i] > 2004) daysSince971207 += 1;
      if (iyear[i] > 2008) daysSince971207 += 1;
      if (iyear[i] > 2012) daysSince971207 += 1;
      if (iyear[i] > 2016) daysSince971207 += 1;
      if (iyear[i] > 2020) daysSince971207 += 1;
      if (iyear[i] > 2024) daysSince971207 += 1;
      if (iyear[i] > 2028) daysSince971207 += 1;
      if (iyear[i] > 2032) daysSince971207 += 1;
      if (iyear[i] > 2036) daysSince971207 += 1;
      if (iyear[i] > 2040) daysSince971207 += 1;

    }  /* end if year is 1997 */

/******************************************************/
/* calculate time in seconds since start of orbit 158 */
/*     - if i = 0, time is start time                 */
/*     - if i = 1, time is end time minus one day and */
/*       tolerance                                    */
/******************************************************/

    secSinceOrb158 =			/* sec from start of orbit 158	*/
	-75277.1			/* to start of day 342		*/
	+ daysSince971207 * 86400.0	/* + sec in days since 12/7/97	*/
	- 300.0;			/* - 300 sec tolerance due to	*/
					/* accuracy of linear fit	*/

/********************************************************************/
/* make end time at the end of the day by adding 86400 seconds (one */
/* day) and add 600.0 seconds to the result to:                     */
/*     1) negate the 300 seconds subtracted above (a bug) and       */
/*     2) provide a +300 second tolerance to the end time           */
/********************************************************************/

    if ( i == 1 ) secSinceOrb158 = secSinceOrb158 + 86400.0 + 600.0;

/*************************************************************/
/* calculate orbit number.  fractional orbit number is based */
/* on a linear fit.                                          */
/*                                                           */
/* note: our post-boost fit is to time since start of 21507  */
/*       versus orbit number, so we use seconds since 21507  */
/*       along with the fit parameters to get the orbit #.   */
/*************************************************************/

    fracOrbnum = (secSinceOrb158 + linFitYint) / linFitSlope;

    if (fracOrbnum > 21500)
    {
      fracOrbnum = (secSinceOrb158 - sec158to21507 - linFitYint2) /
                   linFitSlope2;
    }

/***********************************************************************/
/* reset orbit number to 158 (earliest valid orbit) and set flag for   */
/* orbit number reset to:                                              */
/*                                                                     */
/*   1) for starting orbit number and days since 12/7/97 <= 0, or      */
/*   2) for starting orbit number and days since 12/7/97 <= 0 and      */
/*          ending   orbit number and days since 12/7/97 <  0          */
/*                                                                     */
/* otherwise, set orbit number to integer truncated from fraction      */
/*                                                                     */
/* the main routine errors out if the stopping date is before the      */
/* starting date so the case where the ending orbit number is set to   */
/* 158 but the starting orbit number isn't reset to 158 doesn't exist  */
/***********************************************************************/

    if (i == 0)				/* starting orbit number	*/
    {

      if (daysSince971207 <= 0)
      {

        *numOrbStart = 158;		/* earliest valid orbit is 158	*/

        status = 1;			/* set return flag to reset for	*/
					/* just starting orbit number	*/
      }
      else
      {
        *numOrbStart = (int) fracOrbnum - 1;
      }

    }
    else				/* ending orbit number		*/
    {

      if (daysSince971207 < 0)
      {

        *numOrbEnd = 158;		/* earliest valid orbit is 158	*/

        status = 2;			/* set return flag to reset for	*/
					/* starting and ending orbit	*/
					/* numbers			*/
      }
      else
      {
        *numOrbEnd = (int) fracOrbnum;
      }

    }  /* end ending orbit number */

  }  /* end loop over start and end date */

  return status;

}
