/* CVS Comment Section
 * 
 * $Log: findOrbits.c,v $
 * Revision 1.2  2011/10/19 20:17:28  hensley
 * Added include comOff.h, enhanced error handling
 *
 * Revision 1.1  2008/10/22 21:31:21  hensley
 * First import
 * 
 * ---------------------------------------------------------------------
 * As stored under this directory ^^
 * 
 * As stored under off/web_page/select vv
 * ---------------------------------------------------------------------
 * 
 * Revision 1.2  2008/10/16 22:26:51  hensley
 * Paths set for new server configuration
 * 
 * Revision 1.1.1.1  2008/08/20 23:54:54  hensley
 * First Import
 * 
 * ---------------------------------------------------------------------
 * PPS CVS ^^
 * 
 * TSDIS CVS vv
 * ---------------------------------------------------------------------
 * 
 * Revision 1.7  2002/10/11 22:45:43  hensley
 * Added error handling and debugging code
 * 
 * Revision 1.6  2002/08/12 17:23:51  hensley
 * Fixed CVS header section
 * 
 * Revision 1.5  2001/04/11 19:45:12  hensley
 * Took Out Debug Statements
 * 
 * Revision 1.4  2001/04/11 19:40:49  hensley
 * New Code For Direction Of Pass PLUS Debug Statements
 * 
 * Revision 1.3  2001/02/14 18:58:59  hensley
 * Added pass direction code
 * 
 * Revision 1.2  2001/01/04 19:04:18  hensley
 * Changed maximum orbit number to 30000 from 20000
 * 
 * Revision 1.1.1.1  2000/11/10 23:11:58  hensley
 * Baseline Source Code
 * 
 */

/**************************************************************************
* Program Name:
* ------------
* findOrbits.c
* 
* Description:
* -----------
* Finds information on all overflights near a given latitude/longitude
* location, within a given radius (rounded to 20 km.), for a range of orbit
* numbers for TRMM.
* 
* Routines Called:
* ---------------
* getLonMaxLatA
* timeVSRLon
* PGS_TD_TAItoUTC
* SPAspatialBound
* 
* Return value:
* ------------
* List of orbits that includes orbit number, GMT, MST, direction of the
* swath, longitude of maximum latitude, and the orbit's proximity to the
* selected location
* 
* Input:
* -----
* Latitude, longitude, radius, start and stop orbits
* 
* Output:
* ------
* Radius and start and stop orbits adjusted as needed
* 
* Date		Programmer Name		Change Description
* ----		---------------		------------------
* Spring-98	Steve Bilanow		First written
* 12-JUN-98	Gwyn Fireman		Converted to callable function
* 15-JUL-98	Steve Bilanow		Added overpass time to minute
* 					(new routine timeVSRLon called)
* 					and adjusted lon_of_max_lat fit
* 		Rodney Boyette		Placed output in new structure
* 30-DEC-98	Steve Bilanow		Adjusted fit parameters
* 16-MAR-99	Steve Bilanow		Call new routine, getLonMaxLatA.c,
* 					to get Lon of Max Lat with data fit
* 					using three linear segments
* 09-JUN-99	S.B. & Michael Hensley	Update to get proximity to 20 km
* 					resolution
* 18-MAY-00	S.B. & M.H.		Added local mean solar time,
* 					cleaned up, better variable names
* 25-OCT-00	Michael Hensley		Cleaned up and commented more
* 14-FEB-01	Michael Hensley		Added pass direction code
* 11-APR-01	Michael Hensley		Took out debug statements and
* 					added Steve's pass direction code
* ??-AUG-02	Michael Hensley		Added error handling and debugging
* 					code
* 22-OCT-08	Michael Hensley		Removed error handling, debugging,
* 					emailing, and file output code
* 12-OCT-11	Michael Hensley		Added error handling for returns
* 					from calls to PGS_TD_TAItoUTC and
* 					SPAspatialBound
* 					Reworked includes and prototypes
* 					Enhanced error reporting
**************************************************************************/

/*****************/
/* include files */
/*****************/

#include "comOff.h"

/*****************/
/* routine start */
/*****************/

sOrbitInfo *findOrbits (float lat,		/* latitude		*/
			float lon,		/* longitude		*/
			float *radius,		/* search radius	*/
			int *nOrbStart,		/* starting orbit	*/
			int *nOrbEnd)		/* ending orbit		*/

{

/*****************************/
/* local variable definition */
/*****************************/

  sOrbitInfo *orbitEntry;	/* pointers to linked lists of		*/
  sOrbitInfo *lastOrbitEntry;	/* structures containing overflight	*/
  sOrbitInfo *homeOrbitEntry;	/* information				*/

  float swathw;			/* used by SPAspatialBound		*/
  float latmin;
  float latmax;
  float lonmin;
  float lonmax;
  float ascmin1;
  float ascmax1;
  float desmin1;
  float desmax1;
  float ascmin2;
  float desmin2;

  float ascmin1_save[50];	/* swath intersection criteria for	*/
  float ascmax1_save[50];	/* various swath widths			*/
  float desmin1_save[50];
  float desmax1_save[50];
  float ascmin2_save[50];
  float desmin2_save[50];
  float halfw_save[50];
  float halfwMet;
  float halfw; 
  int ihalfw;

  float lonMaxLat;	/* lon of max latitude for current orbit	*/
  float radiusUsed;	/* input radius is rounded to nearest 20 km.	*/
  float rlon;		/* relative longitude				*/
  float timeFromStart;	/* time from start of orb to site, in min	*/

  float lonMaxLatA[MAX_ORB_NUM];
			/* array of longitudes of max latitudes		*/

  float ascMaxPrev;	/* value of ascmax1 for previous (wider) swath	*/
  float desMaxPrev;	/* value of desmax1 for previous (wider) swath	*/

  double taiAtSite;	/* TAI time, approximate, at site position	*/
  double taiForMST;	/* TAI time for local Mean Solar Time		*/
  double taitime;	/* TAI time in sec since 1/1/93 for orb start	*/

  double taiTimeA[MAX_ORB_NUM];	/* array of TAI times			*/

  int count;			/* search radius step counter		*/
  int countMax;			/* number of search radius steps	*/
  int i;			/* loop counter				*/
  int iflag;			/* flag to show asc or des segment	*/
  int iflag1;			/* flag to show asc or des segment	*/
  int iflagWide;		/* flag to show that overflight occurs	*/
				/* at a certain search radius step	*/
  int iMetStill;		/* flag to show narrowest swath found	*/
  int nOrbit;			/* orbit loop counter			*/
  int nOrbitInit = 158;		/* initial orbit			*/
  int nOrbitMax  = MAX_ORB_NUM;	/* maximum orbit number allowed		*/
  int returnStatus = 0;		/* return status from routine calls and	*/
				/* fprintf				*/
  
  char *dirString = "DD";	/* temp dir. of pass holding string	*/
  char *northEast = "NE";	/* possible directions of passes	*/
  char *northWest = "NW";
  char *southEast = "SE";
  char *southWest = "SW";
  char *blankDirString = "  ";	/* no direction assigned if pass within	*/
				/* 20 km				*/

  char *segNorth  = "Ascending ";	/* asc segment description	*/
  char *segSouth  = "Descending";	/* des segment description	*/
  char *segString = "dummytext1";	/* temp holding string		*/

  char asciiUTC[28];	/* PGS routine output ASCII UTC string		*/
  char asciiGMT[17];	/* ASCII string for GMT date/time output	*/
  char asciiTMP[17];	/* temp ASCII string for date/time output	*/
  char asciiMST[6];	/* ASCII string for local Mean Solar Time (MST)	*/
			/* output					*/

  char *fileID   = "spa_model_params.dat";

/*********************/
/* begin calculation */
/*********************/
  
  homeOrbitEntry = NULL;

  latmax = lat;
  latmin = lat;
  lonmax = lon;
  lonmin = lon;
  
/**************************************************************/
/* determine number of search radius steps (20 km increments) */
/* and actual radius to use (rounded to nearest 20 km)        */
/**************************************************************/

  countMax = (*radius+10.0) / 20.0;

  if (countMax <  1) countMax =  1;
  if (countMax > 40) countMax = 40;

  radiusUsed = countMax * 20.0;

  *radius    = radiusUsed;
  
/********************************************************************/
/* compute and store SPAspatialBound criteria for an array of swath */
/* widths: 20 km swath half width, 40 km half width, ... up to      */
/* input radius (max 800 km)                                        */
/********************************************************************/

  for (count = 1; count <=countMax; count++)
  {

    swathw = count * 40.0;
    halfw  = count * 20.0;
    
    returnStatus = SPAspatialBound (fileID, swathw,
                                    latmin, latmax,
                                    lonmin, lonmax,
                                    &ascmin1, &ascmax1,
                                    &desmin1, &desmax1,
                                    &ascmin2, &desmin2);
    
    switch (returnStatus)
    {

      case 0:
        break;

      case SPA_RETURN1_NEG_SWATH_WIDTH:
        printf("\n\nERROR in file %s, line %d, returned from call",
               __FILE__, __LINE__);
        printf("\nto SPAspatialBound. Negative swath width input.");
        printf("\nSwath width = %f\n\n", swathw);
        exit(EXIT_FAILURE);

      case SPA_RETURN2_MIN_GT_MAX_LAT:
        printf("\n\nERROR in file %s, line %d, returned from call",
               __FILE__, __LINE__);
        printf("\nto SPAspatialBound. Min > Max Latitude.");
        printf("\nMin lat = %f,  Max lat = %f\n\n", latmin, latmax);
        exit(EXIT_FAILURE);

      case SPA_RETURN3_MIN_GT_MAX_LON:
        printf("\n\nERROR in file %s, line %d, returned from call",
               __FILE__, __LINE__);
        printf("\nto SPAspatialBound. Min > Max Longitude.");
        printf("\nMin lon = %f,  Max lon = %f\n\n", lonmin, lonmax);
        exit(EXIT_FAILURE);

      case SPA_RETURN4_BOX_N_OR_S_SWATH:
        printf("\n\nERROR in file %s, line %d, returned from call",
               __FILE__, __LINE__);
        printf(
          "\nto SPAspatialBound. Box N or S of swath boundaries.\n\n");
        exit(EXIT_FAILURE);

      case SPA_RETURN5_DEFAULT_MODEL_PARAMS:
        printf("\n\nWARNING in file %s, line %d, returned from call",
               __FILE__, __LINE__);
        printf(
          "\nto SPAspatialBound. Using default model parameters.\n\n");
        break;

      default:
        printf("\n\nERROR in file %s, line %d, returned from call",
               __FILE__, __LINE__);
        printf("\nto SPAspatialBound. Unknown error code = %d\n\n",
               returnStatus);
        exit(EXIT_FAILURE);

    }

/***********************************/
/* add extra tolerance to criteria */
/***********************************/

    ascmin1_save[count] = ascmin1 - 0.0001;
    ascmax1_save[count] = ascmax1 + 0.0001;
    desmin1_save[count] = desmin1 - 0.0001;
    desmax1_save[count] = desmax1 + 0.0001;
    ascmin2_save[count] = ascmin2 - 0.0001;
    desmin2_save[count] = desmin2 - 0.0001;
    halfw_save[count] =  halfw;

  }  /* end for loop over count */
  
/*******************************************************************/
/* adjust orbit numbers as needed for bad input (don't allow orbit */
/* numbers less than nOrbitInit or greater than nOrbitMax )        */
/*******************************************************************/
  
  if (*nOrbStart < nOrbitInit) *nOrbStart = nOrbitInit;
  if (*nOrbStart > nOrbitMax ) *nOrbStart = nOrbitMax;
  if (*nOrbEnd   > nOrbitMax ) *nOrbEnd   = nOrbitMax;
  if (*nOrbStart > *nOrbEnd  ) *nOrbEnd   = *nOrbStart + 16;
 
/********************************************************************/
/* get array of longitude of maximum latitude values, and TAI times */
/* at orbit start, for the orbit numbers requested.  returnStatus   */
/* is always zero.                                                  */
/********************************************************************/

  returnStatus = getLonMaxLatA(*nOrbStart,
                               *nOrbEnd,
                               lonMaxLatA,
                               taiTimeA);

/********************************************************************/
/* step through orbits and check spatial search criteria.  for each */
/* orbit find the narrowest swath that includes the location.       */
/********************************************************************/
  
  for (nOrbit = *nOrbStart; nOrbit <= *nOrbEnd; nOrbit++)
  {

    lonMaxLat = lonMaxLatA[nOrbit-*nOrbStart];
    taitime   = taiTimeA[nOrbit-*nOrbStart];
    
/***********************************************/
/* first check if widest swath criteria is met */
/***********************************************/
    
    count = countMax;
    ascmin1 =  ascmin1_save[count];
    ascmax1 =  ascmax1_save[count];
    desmin1 =  desmin1_save[count];
    desmax1 =  desmax1_save[count];
    ascmin2 =  ascmin2_save[count];
    desmin2 =  desmin2_save[count];
    halfw   =  halfw_save[count];
    
    iflag = 0;
    
/******************************/
/* ascending segment criteria */
/******************************/

    if ( ( ascmin1 < lonMaxLat
      && lonMaxLat < ascmax1 )
      || ascmin2 < lonMaxLat ) iflag = iflag + 1;

/*******************************/
/* descending segment criteria */
/*******************************/
    
    if ( ( desmin1 < lonMaxLat
      && lonMaxLat < desmax1 )
      || desmin2 < lonMaxLat ) iflag = iflag + 2;
    
    iflagWide = iflag;
    
/*********************************************************************/
/* if widest criteria is met, check more narrow ones (if they exist) */
/* until criteria is not met or most narrow swath is selected        */
/*********************************************************************/
    
    if (iflagWide > 0)
    {

/****************************************************/
/* save flag settings from widest swath check above */
/****************************************************/

      iflag1 = iflag;
      halfwMet  = halfw;
      iMetStill = 1;
      ascMaxPrev = ascmax1;
      desMaxPrev = desmax1;

/***************************************************************/
/* if only one swath width to check, set flag to skip checking */
/* smaller swath widths                                        */
/***************************************************************/

      if (countMax == 1) iMetStill = 0;
      
/***************************************************/
/* while narrower swaths still may contain lat/lon */
/***************************************************/

      while (iMetStill > 0)
      {

        count = count - 1;
        ascmin1 =  ascmin1_save[count];
        ascmax1 =  ascmax1_save[count];
        desmin1 =  desmin1_save[count];
        desmax1 =  desmax1_save[count];
        ascmin2 =  ascmin2_save[count];
        desmin2 =  desmin2_save[count];
        halfw   =  halfw_save[count];

        iflag    = 0;

/******************************/
/* ascending segment criteria */
/******************************/

        if ( ( ascmin1 < lonMaxLat
          && lonMaxLat < ascmax1 )
          || ascmin2 < lonMaxLat ) iflag = iflag + 1;

/*******************************/
/* descending segment criteria */
/*******************************/

        if ( ( desmin1 < lonMaxLat
          && lonMaxLat < desmax1 )
          || desmin2 < lonMaxLat ) iflag = iflag + 2;

/******************************************************************/
/* if narrower criteria is met, set flags to indicate northbound  */
/* or southbound, half width of swath, & save ascmax1 & desmax1   */
/******************************************************************/

        if (iflag > 0)
        {
          iflag1    = iflag;
          halfwMet  = halfw;
          ascMaxPrev = ascmax1;
          desMaxPrev = desmax1;
        }

/**********************************************************/
/* if narrower criteria is no longer met, exit while loop */
/**********************************************************/

        else
        {
          iMetStill = 0;  /* indicates narrowest swath already found */
        }

/***************************************/
/* stop checking after narrowest swath */
/***************************************/

        if (count == 1) iMetStill = 0;

      }  /* end of while loop over narrower swaths */
      
      ihalfw = halfwMet;

/******************************************************************/
/* determine whether the orbit segment is ascending or descending */
/* and the direction of the pass (NW, NE, SW, SE)                 */
/******************************************************************/

      if (iflag1 == 1)  /* ascending/northbound segment */
      {

        segString = segNorth;

/********************************************************************/
/* use the default of NW since with an ascending orbit segment, two */
/* of the three possible triggers to throw the code out of the loop */
/* above:                                                           */
/* 1) ascmin1 crossed lonMaxLat                                     */
/* 2) ascmin2 crossed lonMaxLat                                     */
/* would cause the pass to be NW of the lat/lon input               */
/********************************************************************/

        dirString = northWest;

/**********************************************************************/
/* normal case of decreasing upper bound....                          */
/* If upper (right) side of bound was missed in the last narrowing of */
/* the swath, then the direction must be to the SE                    */
/**********************************************************************/

        if ( (ascMaxPrev - ascmax1) > 0.0 )
        {
          if ( (ascmax1 <= lonMaxLat) && (lonMaxLat <= ascMaxPrev) )
            dirString = southEast;
        }

/************************************************************************/
/* unusual case of upper bound increasing approximately 360 degrees.... */
/* In this case the upper bound was previously close to -180 but now is */
/* close to + 180 degrees.  Thus we check if lonMaxLat was in that      */
/* reduced swath criteria by checking if it is even closer to -180 than */
/* ascMaxPrev, or closer to +180 than the new criteria, ascmax1.        */
/************************************************************************/

        else
        {
          if ( (lonMaxLat <= ascMaxPrev) || (ascmax1 <= lonMaxLat) )
            dirString = southEast;
        }

/********************************************************************/
/* if the lat input is above 35 degrees/below -35 degrees (the TRMM */
/* satellite orbit inclination is 35 degrees), then the pass is     */
/* always south/north of the lat input                              */
/********************************************************************/

        if (lat >  35.0) dirString = southEast;
        if (lat < -35.0) dirString = northWest;

      }
      else  /* descending/southbound segment */
      {

        segString = segSouth;

/********************************************************************/
/* use the default of SW since with a descending orbit segment, two */
/* of the three possible triggers to throw the code out of the loop */
/* above:                                                           */
/* 1) desmin1 crossed lonMaxLat                                     */
/* 2) desmin2 crossed lonMaxLat                                     */
/* would cause the pass to be SW of the lat/lon input               */
/********************************************************************/

        dirString = southWest;

/**********************************************************************/
/* normal case of decreasing upper bound....                          */
/* If upper (right) side of bound was missed in the last narrowing of */
/* the swath, then the direction must be to the NE                    */
/**********************************************************************/

        if ( (desMaxPrev - desmax1) > 0.0 )
        {
          if ( (desmax1 <= lonMaxLat) && (lonMaxLat <= desMaxPrev) )
            dirString = northEast;
        }

/************************************************************************/
/* unusual case of upper bound increasing approximately 360 degrees.... */
/* In this case the upper bound was previously close to -180 but now is */
/* close to + 180 degrees.  Thus we check if lonMaxLat was in that      */
/* reduced swath criteria by checking if it is even closer to -180 than */
/* desMaxPrev, or closer to +180 than the new criteria, desmax1.        */
/************************************************************************/

        else
        {
          if ( (lonMaxLat <= desMaxPrev) || (desmax1 <= lonMaxLat) )
            dirString = northEast;
        } 

/********************************************************************/
/* if the lat input is above 35 degrees/below -35 degrees (the TRMM */
/* satellite orbit inclination is 35 degrees), then the pass is     */
/* always south/north of the lat input                              */
/********************************************************************/

        if (lat >  35.0) dirString = southWest;
        if (lat < -35.0) dirString = northEast;

      }  /* end else descending/southbound segment */
      
      if (ihalfw == 20.0) dirString = blankDirString;

/***************************************************************************/
/* get time of passage over site relative to orbit start time.  timeVSRLon */
/* gives time from the start of the orbit, in minutes, given the longitude */
/* relative to the lon of max lat.                                         */
/***************************************************************************/

      rlon = lon - lonMaxLat;

      if (rlon < -180.0) rlon = rlon + 360.0;
      if (rlon >  180.0) rlon = rlon - 360.0;

      timeFromStart = timeVSRLon(rlon);

      taiAtSite = taitime + timeFromStart * 60.0;

/**************************************************************************/
/* use time conversion utility to convert TAI atomic seconds since 1-1-93 */
/* to a calendar date/time format.  add 30 seconds to round to minute.    */
/**************************************************************************/

      returnStatus = PGS_TD_TAItoUTC(taiAtSite+30.0, asciiUTC);

      /********************/
      /* check for errors */
      /********************/

      if (returnStatus != PGS_S_SUCCESS &&
          returnStatus != PGSTD_W_PRED_LEAPS)
      {
        printf("\n\nERROR in file %s, line %d, when converting TAI",
               __FILE__, __LINE__);
        printf("\ntime to UTCA format - TAI = %s\n\n", taiAtSite+30.0);
        exit(EXIT_FAILURE);
      }
      else
      {
        asciiUTC[10] = 32;
        strncpy(asciiGMT, asciiUTC, 16);  /* copy first 16 characters */
      }

/**********************************************************************/
/* convert to local mean solar time (MST) based on site longitude and */
/* UT (G.M.T.) plus 1 hour for every 15 degrees east to site.  add 30 */
/* seconds to round to minute.                                        */
/**********************************************************************/

      taiForMST = taiAtSite + (3600.0 * lon / 15.0);

      returnStatus = PGS_TD_TAItoUTC(taiForMST+30.0, asciiUTC);

      /********************/
      /* check for errors */
      /********************/

      if (returnStatus != PGS_S_SUCCESS &&
          returnStatus != PGSTD_W_PRED_LEAPS)
      {
        printf("\n\nERROR in file %s, line %d, when converting TAI",
               __FILE__, __LINE__);
        printf("\ntime to UTCA format - TAI = %s\n\n", taiForMST+30.0);
        exit(EXIT_FAILURE);
      }
      else
      {
        strncpy(asciiTMP, asciiUTC, 16);  /* copy first 16 characters */
        for (i=0 ; i<6 ; i++)
        {
          asciiMST[i] = asciiTMP[i+11];  /* take last 6 characters */
        }
      }

/***********************************************/
/* create a new instance of an orbit structure */
/***********************************************/

      orbitEntry = (sOrbitInfo *) malloc(sizeof(sOrbitInfo));

/***********************************************************/
/* load orbit information into the fields of the structure */
/***********************************************************/

      strcpy(orbitEntry->timeGMT, asciiGMT);
      strcpy(orbitEntry->timeMST, asciiMST);
      strcpy(orbitEntry->direction, segString);
      strcpy(orbitEntry->passDir, dirString);

      orbitEntry->orbitNumber = nOrbit;
      orbitEntry->longOfMaxLat = lonMaxLat;
      orbitEntry->proximity = ihalfw;
      orbitEntry->next = NULL;

/***********************************************************************/
/* if the list of orbits is empty, set the address of the first entry. */
/* otherwise, in order to create a NEW last entry, set the address of  */
/* the last entry in the list that points to the next entry            */
/***********************************************************************/

      if (homeOrbitEntry == NULL)
        homeOrbitEntry = orbitEntry;
      else
        lastOrbitEntry->next = orbitEntry;                

/***************************************************/
/* reset the address of the last entry in the list */
/***************************************************/

      lastOrbitEntry = orbitEntry;

    }  /* end of if widest criteria met */

  }  /* end of for loop through orbit numbers */

/****************************************************/
/* return the linked list of structures holding the */
/* overflight information                           */
/****************************************************/

  return homeOrbitEntry;

}
