/* CVS Comment Section
 * 
 * $Log: getLonMaxLatA.c,v $
 * Revision 1.4  2011/12/30 23:18:46  hensley
 * Added ./ in front of the two input file names
 *
 * Revision 1.3  2011/10/19 20:24:32  hensley
 * Added comOff.h, took out long.dat header reading
 *
 * Revision 1.2  2008/10/22 21:54:29  hensley
 * Added hard returns around error messages
 * 
 * Revision 1.1  2008/10/22 21:31:53  hensley
 * First import
 * 
 * ---------------------------------------------------------------------
 * As stored under this directory ^^
 * 
 * As stored under off/web_page/select vv
 * ---------------------------------------------------------------------
 * 
 * Revision 1.2  2008/10/16 22:10:59  hensley
 * Paths set for new server configuration
 * 
 * Revision 1.1.1.1  2008/08/20 23:54:54  hensley
 * First Import
 * 
 * ---------------------------------------------------------------------
 * PPS CVS ^^
 * 
 * TSDIS CVS vv
 * ---------------------------------------------------------------------
 * 
 * Revision 1.7  2004/06/18 20:54:51  hensley
 * Added a debug option to use a test input file
 * 
 * Revision 1.6  2002/10/11 22:25:16  hensley
 * Removed segment fits code/new prediction/error handling/predicted
 * fit data from file
 * 
 * Revision 1.5  2002/08/12 17:22:50  hensley
 * Fixed CVS header section
 * 
 * Revision 1.4  2001/09/27 21:58:48  hensley
 * Enhanced Code to Work at New Orbit Altitude - May Adjust Later
 * 
 * Revision 1.3  2001/05/17 19:24:40  hensley
 * Added a new predicted fit segment
 * 
 * Revision 1.2  2001/02/06 22:22:15  hensley
 * Changed the predicted curve fit parameter values
 * 
 * Revision 1.1.1.1  2000/11/10 23:11:58  hensley
 * Baseline Source Code
 * 
 */

/***********************************************************************
* Function Statement:
* -------------------
* int getLonMaxLatA(	int nOrbStart,
* 			int nOrbEnd,
* 			float lonMaxLatA[],
* 			double taiTimeA[])
* 
* Description:
* -----------
* Returns an array of values for the longitude of maximum latitude and
* an array of orbit start times associated with the input range of orbit
* numbers.
* 
* Input Parameters:
* ----------------
* nOrbStart	first orbit to process
* nOrbEnd	last orbit to process
* 
* Output Parameters:
* -----------------
* lonMaxLatA	array of longitudes of maximum latitude
* taiTimeA	array of orbit start times in TAI format
* 
* Date		Programmer Name		Change Description
* ----		---------------		------------------
* 15-MAR-99	Steve Bilanow		Initial Version
* ??-JUN-99	Michael Hensley		Added read from long.dat for
* 						early orbits
* ??-AUG-99	Michael Hensley		Added 5th & 6th curve fits and
* 					read from long.dat for orbits
* 					through orbit 9811
* 08-SEP-99	Michael Hensley		Changed 5th and 6th curve fits
* 					and made read from long.dat up
* 					to orbit 10237
* 04-JAN-00	Michael Hensley		Added fits 7 & 8 and variable
* 					maxOrbNum
* 10-JAN-00	Steve Bilanow		Adjusted projected fit (#8)
* 					offsets
* 15-SEP-00	Michael Hensley		Added fit 9
* 12-OCT-00	Michael Hensley		Cleaned up and commented more
* 10-NOV-00	Michael Hensley		Took out fit 9 and made fit 8
* 					the predicted fit again
* 06-FEB-01	Michael Hensley		Changed the predicted curve fit
* 					parameter values
* 17-MAY-01	Michael Hensley		Added new predicted fit segment
* 08-AUG-01     Steve Bilanow           Making estimates for new orbit
* 12-AUG-02	Michael Hensley		Stripped out fit segments code,
* 					added new prediction, added
* 					error message emailing, added
* 					calls to errorPage, added read
* 					in predicted fit data from file,
* 					other misc.
* 22-OCT-08	Michael Hensley		Replaced error emails with error
* 					reporting to standard out
* 12-OCT-11	Michael Hensley		Reworked include and prototype
* 					statements
* 					Took out code to read long.dat
* 					file 2-line header
* 					Enhanced header comments
* 29-DEC-11	Michael Hensley		Added ./ in front of the two
* 					input file names
***********************************************************************/

/*****************/
/* include files */
/*****************/

#include "comOff.h"

/*****************/
/* routine start */
/*****************/

int getLonMaxLatA(	int nOrbStart,
			int nOrbEnd,
			float lonMaxLatA[],
			double taiTimeA[])

{

/******************************************************/
/* variable declarations and constants from data fits */
/******************************************************/

  FILE *SPAORB_input;		/* orbit trend data input file handle	*/
  FILE *PREDFIT_input;		/* predicted fit data input file handle	*/

  int status=0;			/* return status, default zero		*/
  int count=0;			/* number of lines in input file	*/
  int i;			/* array counter			*/

  int nOrb;			/* counter for loop through orbits	*/
  int nOrbindex;		/* array index for filling output array	*/
  int maxOrbNum;		/* high orbit number in "long.dat" file	*/

  int predStartOrb;		/* predicted fit starting orbit number	*/

  int ierror=0;			/* error code returned from calls	*/
  int ierror1=0;		/* error code returned from fscanf	*/
  int ierror2=0;		/* error code returned from fscanf	*/
  int ierror3=0;		/* error code returned from fscanf	*/
  int ierror4=0;		/* error code returned from fscanf	*/
  int ierror5=0;		/* error code returned from fscanf	*/

  int orbitNumber[MAX_ORB_NUM];	/* array of orb #s read from "long.dat"	*/
  int elapsedTime[MAX_ORB_NUM];	/* array of elapsed times read from	*/
				/* "long.dat"				*/

  double secPerOrb;		/* sec/orbit for predicted fit		*/
  double lonPerOrb;		/* deg lon/orbit for predicted fit	*/
  double predStartSec;		/* predicted fit starting orbit start	*/
				/* time (TAI)				*/
  double predStartLon;		/* predicted fit starting orbit		*/
				/* longitudes of maximum latitude	*/

  double secSinceOrb158;	/* sec since start of orbit 158 for	*/
				/* input date				*/
  double lonMaxLatD;		/* lon of maximum lat in degrees	*/

  double secToOrb158 = 155681681.1;	/* sec from TAI ref start to	*/
					/* orbit 158 start at		*/
					/* 1997-12-07T20:54:37.1	*/

  double loml[MAX_ORB_NUM];	/* array to hold longitudes of maximum	*/
				/* latitude read from "long.dat"	*/

  char junkString[100];		/* junk to be ignored			*/

  char *fg_p1;			/* pointer to return from fgets		*/
  char *fg_p2;			/* pointer to return from fgets		*/

  char *errText1 = "when reading header lines of file, long.dat";
  char *errText2 = "when reading data line";

/*************************************************************/
/* open the predicted fit data file or if the file cannot be */
/* opened, report the error and exit                         */
/*************************************************************/

  if ( ( PREDFIT_input = fopen("./pred_fit.dat", "r") ) == NULL )
  {

/****************************************/
/* print error to standard out and exit */
/****************************************/

    printf(
"\n\nERROR in file %s, line %d, when opening file, pred_fit.dat\n\n",
    __FILE__, __LINE__);

    exit(EXIT_FAILURE);

  }
  else  /* file opened successfully */
  {

/****************/
/* read in data */
/****************/

    ierror1 = fscanf(PREDFIT_input, "%d" , &predStartOrb);
    ierror2 = fscanf(PREDFIT_input, "%lf", &secPerOrb);
    ierror3 = fscanf(PREDFIT_input, "%lf", &predStartSec);
    ierror4 = fscanf(PREDFIT_input, "%lf", &lonPerOrb);
    ierror5 = fscanf(PREDFIT_input, "%lf", &predStartLon);

/**********************************************************/
/* if an error occurred while reading in the data, report */
/* the error, close the input file, and exit              */
/**********************************************************/

    if ( ierror1 != 1 ||
         ierror2 != 1 ||
         ierror3 != 1 ||
         ierror4 != 1 ||
         ierror5 != 1 )
    {

/***********************************************************/
/* print error to standard out, close input file, and exit */
/***********************************************************/

      printf(
"\n\nERROR in file %s, line %d, when reading file, pred_fit.dat\n\n",
      __FILE__, __LINE__);

      fclose(PREDFIT_input);

      exit(EXIT_FAILURE);

    }  /* end if a read error */

  } /* end else file opened successfully */

/********************/
/* close input file */
/********************/

  fclose(PREDFIT_input);

/*************************************************************/
/* open the orbit data file or if the file cannot be opened, */
/* report the error and exit                                 */
/*************************************************************/

  if ( ( SPAORB_input = fopen("./long.dat", "r") ) == NULL )
  {

/****************************************/
/* print error to standard out and exit */
/****************************************/

    printf(
    "\n\nERROR in file %s, line %d, when opening file, long.dat\n\n",
    __FILE__, __LINE__);

    exit(EXIT_FAILURE);

  }
  else  /* file opened successfully */
  {

/*******************************************/
/* loop to read the input file into arrays */
/*******************************************/

    while ( !feof(SPAORB_input) )
    {

/**************************************/
/* parse line of data from input file */
/**************************************/

      ierror = fscanf(SPAORB_input,"%13d%13d%lf",
				   &orbitNumber[count],
				   &elapsedTime[count],
				   &loml[count]);

/************************************************************/
/* if there is an fscanf error, report the error, close the */
/* input file, and exit                                     */
/************************************************************/

      if ( ierror != 3 && ierror != EOF )
      {

/***********************************************************/
/* print error to standard out, close input file, and exit */
/***********************************************************/

        printf(
        "\n\nERROR in file %s, line %d, %s %d of file, long.dat\n\n",
        __FILE__, __LINE__, errText2, count+1);

        fclose(SPAORB_input);

        exit(EXIT_FAILURE);

      }  /* end if data line parsing error */

      count++;

    }  /* end while loop to read the input file into arrays */

  }  /* end else file opened successfully */

  maxOrbNum = orbitNumber[count-2];

/********************************************************************/
/* loop through orbits and read in or calculate seconds since orbit */
/* 158 and the longitude of maximum latitude in degrees             */
/********************************************************************/

  for ( nOrb=nOrbStart ; nOrb<nOrbEnd+1 ; nOrb++ )
  { 

    if ( nOrb <= maxOrbNum )  /* read definitive data from input file */
    {

      for ( i=0 ; i<count ; i++ )  /* loop through input file data */
      {

        if ( nOrb == orbitNumber[i] )  /* found orbit in input data */
        {

          secSinceOrb158 = elapsedTime[i];
          lonMaxLatD     = loml[i];
          i              = count; 

        }  /* end if */

      }  /* end for */

    }  /* end if */

/*****************************************************************/
/* else orbit number > maxOrbNum so use predicted fit instead of */
/* input file                                                    */
/*****************************************************************/

    else
    {

      secSinceOrb158 = (nOrb-predStartOrb)*secPerOrb + predStartSec
                                                     - secToOrb158;
      lonMaxLatD     = (nOrb-predStartOrb)*lonPerOrb + predStartLon; 

    }  /* end else */

/********************************************************************/
/* find the time since the start of orbit 158 in TAI time and store */
/********************************************************************/

    nOrbindex = nOrb - nOrbStart;
    taiTimeA[nOrbindex] = secSinceOrb158 + secToOrb158;

/**********************************************/
/* convert longitude to -180 to 180 and store */
/**********************************************/

    lonMaxLatD = fmod(lonMaxLatD, 360.0);
    if (lonMaxLatD < -180.0) lonMaxLatD = lonMaxLatD + 360.0;
    if (lonMaxLatD >  180.0) lonMaxLatD = lonMaxLatD - 360.0;
    lonMaxLatA[nOrbindex] = (float)lonMaxLatD;

  }  /* end loop over orbits */

/********************/
/* close input file */
/********************/

  fclose(SPAORB_input);

  return status;

}
