/* CVS Comment Section
 * 
 * $Log: mainFindOrbits.c,v $
 * Revision 1.5  2011/10/19 20:13:13  hensley
 * Added findOrbNums, input via command line args
 *
 * Revision 1.4  2011/08/27 00:08:57  hensley
 * Changed suggested orbit number range
 * 
 * Revision 1.3  2011/07/01 22:40:51  hensley
 * Changed TSDIS to TRMM/PPS
 * 
 * Revision 1.2  2008/10/22 21:37:45  hensley
 * Changed call to findOrbits
 * 
 * Revision 1.1.1.1  2008/08/20 23:54:54  hensley
 * First Import
 * 
 * ---------------------------------------------------------------------
 * PPS CVS ^^
 * 
 * TSDIS CVS vv
 * ---------------------------------------------------------------------
 * 
 * Revision 1.2  2001/04/17 20:41:53  hensley
 * Added Direction To Ground Track Column
 * 
 * Revision 1.1.1.1  2000/11/10 23:11:57  hensley
 * Baseline Source Code
 * 
 */

/***********************************************************************
* Program Name:
* ------------
* FindOrbit_exe
* 
* Description:
* -----------
* UNIX command line version of the TRMM overflight finder code.
* 
* Routines Called:
* ---------------
* findOrbits - calculates overflight data
* 
* Date		Programmer Name		Change Description
* ----		---------------		------------------
* 		Steve Bilanow		First written
* 		Michael Hensley		Many changes
* 04-OCT-00	Michael Hensley		Added comments
* 22-OCT-08	Michael Hensley		Changed call to findOrbits
* 14-OCT-11	Michael Hensley		Added call to findOrbNums
* 					Changed input to command line
* 					Reworked include statements and
* 					removed prototype statements
* 					Added limit checks on inputs
* 					Added checks for error returns
* 					from sub-routines
* 					Added check for stop date before
* 					start date
***********************************************************************/

/*****************/
/* include files */
/*****************/

#include "comOff.h"

/*****************/
/* routine start */
/*****************/

int main(int argc, char *argv[])
{

   /*****************************/
   /* local variable definition */
   /*****************************/

   sOrbitInfo *homeOrbitEntry;
			/* structure to hold overflight data		*/

   sOrbitInfo *tempHomeOrbitEntry;
			/* temporary structure to hold overflight data	*/

   int errorCode;	/* error return code from call to findOrbNums	*/

   int i;		/* number of overflights found			*/

   int inDay[2];	/* start and end day				*/
   int inMonth[2];	/* start and end month				*/
   int inYear[2];	/* start and end year				*/

   int startOrbit;	/* first orbit to search for overflights	*/
   int stopOrbit;	/*  last orbit to search for overflights	*/

   float lat;		/* latitude					*/
   float lon;		/* longitude					*/

   float radius;	/* radius about lat/lon pair			*/

   double startTAI;	/* start date in TAI format			*/
   double  stopTAI;	/*  stop date in TAI format			*/

   char startUTCA[28];	/* start date in UTCA format			*/
   char  stopUTCA[28];	/*  stop date in UTCA format			*/

   /***************************************/
   /* check command line input for errors */
   /***************************************/

   if (argc != 10)
   {
      printf("\n\nUsage: <exe> <start year> <start month> <start day> \\");
      printf("\n             <stop year>  <stop month>  <stop day> \\");
      printf("\n             <latitude>   <longitude>   <search radius>");
      printf("\n");
      printf("\n - Year, month, and day input as integer");
      printf("\n - Start date must be >= December 7, 1997");
      printf("\n - Stop date must be >= start date");
      printf("\n - Start and stop year input in YYYY format");
      printf("\n - Start and stop month from 1 to 12");
      printf("\n - Start and stop day from 1 to 31");
      printf("\n - Overflights are found from 00:00:00 of the start date");
      printf("\n   to 23:59:59.999999 of the stop date");
      printf("\n - Latitude and longitude input in degrees (float)");
      printf("\n - Latitude from -90.0 to 90.0");
      printf("\n - Longitude from -360.0 to 360.0");
      printf("\n - Search radius is a maximum range from the lat/lon");
      printf("\n   in kilometers from 20 to 800 km rounded to the");
      printf("\n   nearest 20 km (float)");
      printf("\n - Example command line:");
      printf("\n   <exe> 2011 11 15 2011 11 30 30.5 -80.5 780");
      printf("\n - Make sure the files long.dat, pred_fit.dat, and");
      printf("\n   spa_model_params.dat are in the same directory");
      printf("\n   where you are running this executable");
      printf("\n\n");
      exit(EXIT_FAILURE);
   }

   /**************************************/
   /* save input data to local variables */
   /**************************************/

   inYear[0]  = atoi(argv[1]);
   inMonth[0] = atoi(argv[2]);
   inDay[0]   = atoi(argv[3]);

   inYear[1]  = atoi(argv[4]);
   inMonth[1] = atoi(argv[5]);
   inDay[1]   = atoi(argv[6]);

   lat        = atof(argv[7]);
   lon        = atof(argv[8]);

   radius     = atof(argv[9]);

   /****************************************************/
   /* check that years are 4-digit and within the TRMM */
   /* mission (1997-present)                           */
   /****************************************************/

   if ( inYear[0] < 1997 || inYear[1] < 1997 )
   {
      printf("\n\nERROR: Year should be 4-digit and in the range");
      printf("\n1997-present.  Start year = %d,  Stop year = %d\n\n",
             inYear[0], inYear[1]);
      exit(EXIT_FAILURE);
   }

   /*******************************************/
   /* check that months are in the range 1-12 */
   /*******************************************/

   if ( inMonth[0] < 1 || inMonth[0] > 12 ||
        inMonth[1] < 1 || inMonth[1] > 12 )
   {
      printf("\n\nERROR: Month should be in the range 1-12");
      printf("\nStart month = %d,  Stop month = %d\n\n",
             inMonth[0], inMonth[1]);
      exit(EXIT_FAILURE);
   }

   /*****************************************/
   /* check that days are in the range 1-31 */
   /*****************************************/

   if ( inDay[0] < 1 || inDay[0] > 31 ||
        inDay[1] < 1 || inDay[1] > 31 )
   {
      printf("\n\nERROR: Day should be in the range 1-31");
      printf("\nStart day = %d,  Stop day = %d\n\n",
             inDay[0], inDay[1]);
      exit(EXIT_FAILURE);
   }

   /*****************************************************/
   /* check that latitude is in the range -90.0 to 90.0 */
   /*****************************************************/

   if (lat < -90.0 || lat > 90.0)
   {
      printf(
         "\n\nERROR: Latitude should be in the range -90.0 to 90.0");
      printf("\nLatitude = %f\n\n", lat);
      exit(EXIT_FAILURE);
   }

   /********************************************************/
   /* check that longitude is in the range -360.0 to 360.0 */
   /********************************************************/

   if (lon < -360.0 || lon > 360.0)
   {
      printf(
         "\n\nERROR: Longitude should be in the range -360.0 to 360.0");
      printf("\nLongitude = %f\n\n", lon);
      exit(EXIT_FAILURE);
   }

   /**************************************************/
   /* check that radius is in the range 0.0 to 800.0 */
   /**************************************************/

   if (radius < 0.0 || radius > 800.0)
   {
      printf(
         "\n\nERROR: Radius should be in the range 0.0 to 800.0");
      printf("\nRadius = %f\n\n", radius);
      exit(EXIT_FAILURE);
   }

   /****************************************************/
   /* create UTCA string representing input start date */
   /* at 00:00:00                                      */
   /****************************************************/

   sprintf(startUTCA, "%4d-%02d-%02dT00:00:00.000000Z",
                      inYear[0], inMonth[0], inDay[0]);

   /*****************************************************/
   /* convert start date to TAI format from UTCA format */
   /*****************************************************/

   errorCode = PGS_TD_UTCtoTAI(startUTCA, &startTAI);

   /********************/
   /* check for errors */
   /********************/

   if (errorCode != PGS_S_SUCCESS &&
       errorCode != PGSTD_W_PRED_LEAPS)
   {
      printf(
           "\nStart date UTCA to TAI conversion failed - UTCA = %s\n\n",
           startUTCA);
      exit(EXIT_FAILURE);
   }

   /***********************************************************/
   /* create UTCA string representing input stop date at 1am. */
   /* set time to 1am so if start & stop dates are the same,  */
   /* this will ensure the stop date in TAI time will be      */
   /* greater than the start date in TAI time.  if the start  */
   /* and stop dates and times were both the same, the start  */
   /* date TAI time may still be greater than the stop date   */
   /* TAI time instead of the two being equal to each other   */
   /* due to round off error.                                 */
   /***********************************************************/

   sprintf(stopUTCA, "%4d-%02d-%02dT01:00:00.000000Z",
                     inYear[1], inMonth[1], inDay[1]);

   /****************************************************/
   /* convert stop date to TAI format from UTCA format */
   /****************************************************/

   errorCode = PGS_TD_UTCtoTAI(stopUTCA, &stopTAI);

   /********************/
   /* check for errors */
   /********************/

   if (errorCode != PGS_S_SUCCESS &&
       errorCode != PGSTD_W_PRED_LEAPS)
   {
      printf(
            "\nStop date UTCA to TAI conversion failed - UTCA = %s\n\n",
            stopUTCA);
      exit(EXIT_FAILURE);
   }

   /********************************************************/
   /* check to see if the input start date is greater than */
   /* the input stop date and if so, issue error and exit  */
   /********************************************************/

   if (startTAI > stopTAI)
   {
      printf("\n\nERROR: Start date greater than stop date!");
      printf("\nStart date = %d/%d/%d,  Stop date = %d/%d/%d.\n\n",
          inMonth[0], inDay[0], inYear[0],
          inMonth[1], inDay[1], inYear[1]);
      exit(EXIT_FAILURE);
   }

   /**************************************************************/
   /* set month to be in the range 0-11 for input to findOrbNums */
   /**************************************************************/

   inMonth[0] = inMonth[0] - 1;
   inMonth[1] = inMonth[1] - 1;

   /*************************************************************/
   /* find orbit numbers of orbits occurring with in date range */
   /*************************************************************/

   errorCode = findOrbNums(inYear, inMonth, inDay, &startOrbit, &stopOrbit);

   /**********************************/
   /* check findOrbNums return codes */
   /**********************************/

   switch (errorCode)
   {

      case 0:		/* nominal					*/

         break;

      case 1:		/* starting orbit number reset to 158		*/

         printf("\n\nStart date before or day of mission start");
         printf("\n(12/7/97) so starting orbit number set to 158");
         printf("\n(first official orbit of the mission)\n\n\n\n");
         break;

      case 2:		/* starting & ending orbit number reset to 158	*/

         printf("\n\nStart date before or day of mission start");
         printf("\n(12/7/97) and end date before mission start so");
         printf("\nstarting orbit number and ending orbit number set");
         printf("\nto 158 (first official orbit of the mission)\n\n\n\n");
         break;

      default:		/* erroneous return code			*/

         printf("\n\nErroneous return from call to findOrbNums!!!");
         printf("\nValid return codes = 0-2,  Return code = %d\n\n",
                errorCode);
         exit(EXIT_FAILURE);

   }

   /*************************/
   /* calculate overflights */
   /*************************/

   homeOrbitEntry = findOrbits(lat, lon, &radius, &startOrbit, &stopOrbit);

   /****************************/
   /* print header for results */
   /****************************/

   printf("\n\n OVERFLIGHTS CALCULATED FOR TRMM/PPS ORBIT NUMBERS");
   printf("\n -------------------------------------------------\n");
   printf("\n  For    Ground Track");
   printf("\n Orbit      Range      In Orbit   Direction to   Approximate UT    Local Mean");
   printf("\n Number  (+/- 20 km)   Segment    Ground Track    Date & Time      Solar Time");
   printf("\n  -----    -------    ----------  ------------  ----------------   ----------");

   /***************************/
   /* variable initialization */
   /***************************/

   i=0;
   tempHomeOrbitEntry = NULL;
   tempHomeOrbitEntry = homeOrbitEntry;

   /***********************************************************/
   /* loop over all overflights and print out overflight data */
   /***********************************************************/

   while (tempHomeOrbitEntry != NULL)
   {

      printf("\n  %5d    %7d    %s       %s       %s     %s",
             tempHomeOrbitEntry -> orbitNumber,
             tempHomeOrbitEntry -> proximity,
             tempHomeOrbitEntry -> direction,
             tempHomeOrbitEntry -> passDir,
             tempHomeOrbitEntry -> timeGMT,
             tempHomeOrbitEntry -> timeMST);

      tempHomeOrbitEntry = tempHomeOrbitEntry -> next;

      i++;

   }  /* end while loop */

   /*********************************/
   /* print overflight summary data */
   /*********************************/

   printf(
         "\n\n%d overflights were found within %.0f km of the selected",
         i, radius);
   printf("\nlocation at latitude = %f and longitude = %f",
          lat, lon);
   printf("\nfor the requested date range %d/%d/%d to %d/%d/%d.\n\n",
          inMonth[0]+1, inDay[0], inYear[0],
          inMonth[1]+1, inDay[1], inYear[1]);

   exit(EXIT_SUCCESS);

}
